import json
import os
import time
import base64
import requests
from PIL import Image
import io
Image.MAX_IMAGE_PIXELS = None

# Configuration
MODEL_NAME = "gpt-5"
API_KEY = os.getenv("OPENAI_API_KEY", "YOUR_OPENAI_API_KEY_HERE")
API_URL = "https://api.openai.com/v1/chat/completions"
DATASET_PATH = "../dataset/dataset_gpt/"
INPUT_JSON = "e5v_results.json"
OUTPUT_JSON = "gpt5_results.json"

# File extensions
IMAGE_EXTENSIONS = ['.jpg', '.jpeg', '.png', '.gif']
MEDIA_EXTENSIONS = ['.mp4', '.jpg', '.png', '.wav', '.mp3', '.pdf', '.jpeg', '.gif', '.txt']
SUPPORTED_EXTENSIONS = ['', '.mp4', '.jpg', '.png', '.jpeg']

# Image processing configuration
MAX_IMAGE_SIZE = (1024, 1024)
RESIZE_METHOD = Image.LANCZOS
JPEG_QUALITY = 85
DEFAULT_IMAGE_FORMAT = "PNG"

# API configuration
GENERATION_TEMPERATURE = 0.4
RATE_LIMIT_DELAY = 1
MAX_RETRIEVED_ITEMS = 5

# MIME type mappings
MIME_TYPE_MAP = {
    "JPEG": "image/jpeg",
    "PNG": "image/png",
    "GIF": "image/gif"
}

print(f"Model: {MODEL_NAME}")
print(f"Dataset: {DATASET_PATH}")
print(f"Input: {INPUT_JSON}")
print(f"Output: {OUTPUT_JSON}")

# File handling utilities
def find_media_resources(filename, base_dir=DATASET_PATH):
    """Find media resources for filename, supporting files and folders."""
    import re
    base_name = re.sub(r'\.(mp4|jpg|png|wav|mp3|pdf|jpeg|gif|txt)$', '', filename, flags=re.IGNORECASE)

    folder_path = os.path.join(base_dir, base_name)
    if os.path.isdir(folder_path):
        return get_all_images_from_folder(folder_path)

    for ext in SUPPORTED_EXTENSIONS:
        file_path = os.path.join(base_dir, base_name + ext)
        if os.path.isfile(file_path):
            return [file_path]

    return []

def get_all_images_from_folder(folder_path):
    """Return all image files from folder, sorted by filename."""
    image_files = []
    for name in os.listdir(folder_path):
        path = os.path.join(folder_path, name)
        if os.path.isfile(path):
            ext = os.path.splitext(name)[1].lower()
            if ext in IMAGE_EXTENSIONS:
                image_files.append(path)
    return sorted(image_files)

# Image processing utilities
def resize_image(img_path, max_size=MAX_IMAGE_SIZE):
    """Resize image while preserving format. Returns (bytes, mime_type) or None."""
    try:
        with Image.open(img_path) as img:
            ext = os.path.splitext(img_path)[1].lower()
            if img.format:
                target_format = img.format
            elif ext in (".jpg", ".jpeg"):
                target_format = "JPEG"
            elif ext == ".png":
                target_format = "PNG"
            elif ext == ".gif":
                target_format = "GIF"
            else:
                target_format = DEFAULT_IMAGE_FORMAT

            img.thumbnail(max_size, RESIZE_METHOD)

            if target_format.upper() == "JPEG" and img.mode not in ("RGB", "L"):
                img = img.convert("RGB")

            buffer = io.BytesIO()
            if target_format.upper() == "JPEG":
                img.save(buffer, format="JPEG", quality=JPEG_QUALITY)
            else:
                img.save(buffer, format=target_format)
            buffer.seek(0)

            mime_type = MIME_TYPE_MAP.get(target_format.upper(), "image/png")
            return buffer.getvalue(), mime_type
    except Exception as e:
        print(f"Error processing {img_path}: {e}")
        return None

# API interaction
def query_gpt5(question, image_paths):
    headers = {
        "Authorization": f"Bearer {API_KEY}",
        "Content-Type": "application/json"
    }

    content = [{"type": "text", "text": f"Question: {question}\nPlease answer based on the images provided."}]

    for img_path in image_paths:
        if img_path:
            print(f"Processing image: {img_path}")
            out = resize_image(img_path)
            if out:
                image_data, mime_type = out
                base64_image = base64.b64encode(image_data).decode('utf-8')
                content.append({
                    "type": "image_url",
                    "image_url": {
                        "url": f"data:{mime_type};base64,{base64_image}"
                    }
                })

    payload = {
        "model": MODEL_NAME,
        "messages": [{"role": "user", "content": content}],
        "temperature": GENERATION_TEMPERATURE
    }

    try:
        response = requests.post(API_URL, headers=headers, json=payload)
        response.raise_for_status()
        return response.json()["choices"][0]["message"]["content"]
    except Exception as e:
        print(f"API error: {e}")
        if hasattr(e, 'response') and hasattr(e.response, 'text'):
            print(f"Response: {e.response.text}")
        return f"Error: {str(e)}"

# Main processing pipeline
def main():
    with open(INPUT_JSON, 'r') as f:
        data = json.load(f)

    results = []

    for i, item in enumerate(data['results']):
        print(f"Processing item {i+1}/{len(data['results'])}: {item['question']}")

        question = item['question']
        positive = item['positive']
        top_5_retrieved = item['top_5_retrieved']
        answer = item['answer']

        image_paths = []
        for file in top_5_retrieved[:MAX_RETRIEVED_ITEMS]:
            media_paths = find_media_resources(file, DATASET_PATH)
            for p in media_paths:
                if os.path.splitext(p)[1].lower() in IMAGE_EXTENSIONS:
                    image_paths.append(p)

        if image_paths:
            response = query_gpt5(question, image_paths)
        else:
            response = "No images found for this question."

        result = {
            "question": question,
            "positive": positive,
            "top_5_retrieved": top_5_retrieved,
            "answer": answer,
            "response": response
        }
        results.append(result)

        with open(OUTPUT_JSON, 'w') as f:
            json.dump(results, f, indent=2)

        time.sleep(RATE_LIMIT_DELAY)

    print(f"Processing complete. Results saved to {OUTPUT_JSON}")

if __name__ == "__main__":
    main()
