# coding=utf-8
# Copyright 2021 The Neural Sddp Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Learn from solved sddp instances."""

from absl import app
from absl import flags
import random
import numpy as np
from typing import Sequence
import functools
import jax
import jax.numpy as jnp
from cond_piecewise_nn import CondPiecewiseNN


FLAGS = flags.FLAGS
flags.DEFINE_integer('seed', 1, 'random seed')
flags.DEFINE_string('loss_type', 'emd', 'mse/emd')


@functools.partial(jax.jit, static_argnums=0)
def jit_apply(model, params, feat, stage):
  return model.apply(params, feat, stage)


@functools.partial(jax.jit, static_argnums=0)
def eval_loss(model, params, feat, stage, target):
  pred_params = jit_apply(model, params, feat, stage)
  if FLAGS.loss_type == 'emd':
    dist = model.emd_approx(pred_params, target)
  elif FLAGS.loss_type == 'mse':
    dist = model.mse(pred_params, target)
  else:
    raise ValueError('unknown loss type %s' % FLAGS.loss_type)
  return jnp.mean(dist)


@functools.partial(jax.jit, static_argnums=0)
def train_step(model, feat, stage, target, optimizer):

  def loss(params):
    return eval_loss(model, params, feat, stage, target)
  l, grads = jax.value_and_grad(loss)(optimizer.target)
  optimizer = optimizer.apply_gradient(grads)
  return l, optimizer


def main(argv):
  del argv
  key = jax.random.PRNGKey(FLAGS.seed)
  np.random.seed(FLAGS.seed)
  random.seed(FLAGS.seed)
  num_vars = 310
  num_pieces = 64
  model = CondPiecewiseNN(num_vars=num_vars,
                          num_stages=5,
                          hidden_size=128,
                          num_pieces=num_pieces,
                          num_layers=1)
  dummy_input = jnp.ones([1, 10], dtype=jnp.float32)  # input feature
  dummy_out, params = model.init_with_output(key, dummy_input,
                                             jnp.array([0], dtype=jnp.int32))
  # neural network predicts the cutting planes
  assert dummy_out.shape[1] == num_pieces and dummy_out.shape[2] == num_vars + 1
  del params


if __name__ == '__main__':
  app.run(main)
