import numpy as np
import random
import scipy.io as spio
import subprocess
import os

""" Define data structure """
class StimulusSpecificData:
    def __init__(self, name):
        self.stimName = name
        
class DataContainer(object):
    def __init__(self):
        pass
            
class AttrDict(dict):
    """dot.notation access to dictionary attributes"""
    __getattr__ = dict.get
    __setattr__ = dict.__setitem__
    __delattr__ = dict.__delitem__

def set_random_seed(seed=1234):
    np.random.seed(seed)
    random.seed(seed)

class Ensembles:
    def __init__(self):
        self.esmbl_registered = {}
    
    def esmbl(self, esmblID):
        attr = 'esmbl{:d}'.format(esmblID)
        return getattr(self, attr)
    
    def setup_esmbl(self, esmblID, obj):
        attr = 'esmbl{:d}'.format(esmblID)
        setattr(self, attr, obj)
        self.esmbl_registered[attr] = obj.name
    
    def esmbl_exist(self, esmblID):
        attr = 'esmbl{:d}'.format(esmblID)
        return hasattr(self, attr)

    def get_registered_esmbls(self):
        return list(self.esmbl_registered.keys())

class bcolors:
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKCYAN = '\033[96m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'

class ModelHook:
    def __init__(self, model, hook_layers_dict, recursive_layers=[]):
        self.model = model
        self.hook_layers_dict = hook_layers_dict    
        self.status = False # hook status
        self.layers_are_recursive = recursive_layers
        self.layers_not_recursive = np.setdiff1d(list(self.hook_layers_dict.keys()),
                                                 recursive_layers).tolist()
            
    def _get_activation(self, layer_name):
        def hook(model, input, output):
            if layer_name in self.layers_are_recursive:
                if self.activation[layer_name] is None:
                    self.activation[layer_name] = []
                self.activation[layer_name].append(output.detach().cpu().numpy())
            else:
                self.activation[layer_name] = output.detach().cpu().numpy()
        return hook
    
    def hook_model(self):
        self.hook_hdl = []
        self.activation = {}
        for layer_name, layer in self.hook_layers_dict.items(): 
            self.hook_hdl.append(
                layer.register_forward_hook(self._get_activation(layer_name))
            )
            self.activation[layer_name] = None
        s = 'hook layer: '
        s += ', '.join([layer_name for layer_name in self.hook_layers_dict.keys()])
        print(s)
        self.status = True
        
    def clear_activation_buffer(self):
        # should be called in every model inference loop
        for layer_name, layer in self.hook_layers_dict.items(): 
            self.activation[layer_name] = None
        
    def update_hooks(self, hook_layers_dict):
        self.unhook_model()
        self.hook_layers_dict = hook_layers_dict
            
    def unhook_model(self):
        if self.status:
            s = 'unhook layer: '
            s += ', '.join([layer_name for layer_name in self.hook_layers_dict.keys()])
            print(s)
            [hdl.remove() for hdl in self.hook_hdl]
            self.status = False
    
    def get_activation(self, layer_name):
        if layer_name in self.layers_are_recursive:
            return np.stack(self.activation[layer_name], axis=0)
        else:
            return self.activation[layer_name]
    
    def get_hooked_layer_name(self):
        return list(self.hook_layers_dict.keys())

def loadmat(filename):
    '''
    this function should be called instead of direct spio.loadmat
    as it cures the problem of not properly recovering python dictionaries
    from mat files. It calls the function check keys to cure all entries
    which are still mat-objects
    '''
    def _check_keys(d):
        '''
        checks if entries in dictionary are mat-objects. If yes
        todict is called to change them to nested dictionaries
        '''
        for key in d:
            if isinstance(d[key], spio.matlab.mat_struct):
                d[key] = _todict(d[key])
            if isinstance(d[key], np.ndarray):
                d[key] = _tolist(d[key])
        return d

    def _todict(matobj):
        '''
        A recursive function which constructs from matobjects nested dictionaries
        '''
        d = {}
        for strg in matobj._fieldnames:
            elem = matobj.__dict__[strg]
            if isinstance(elem, spio.matlab.mat_struct):
                d[strg] = _todict(elem)
            elif isinstance(elem, np.ndarray):
                d[strg] = _tolist(elem)
            else:
                d[strg] = elem
        return d

    def _tolist(ndarray):
        '''
        A recursive function which constructs lists from cellarrays
        (which are loaded as numpy ndarrays), recursing into the elements
        if they contain matobjects.
        '''
        elem_list = []
        for sub_elem in ndarray:
            if isinstance(sub_elem, spio.matlab.mat_struct):
                elem_list.append(_todict(sub_elem))
            elif isinstance(sub_elem, np.ndarray):
                elem_list.append(_tolist(sub_elem))
            else:
                elem_list.append(sub_elem)
        return elem_list
    
    data = spio.loadmat(filename, struct_as_record=False, squeeze_me=True)
    return _check_keys(data)

def nansumwrapper(a, **kwargs):
    if np.isnan(a).all():
        return np.nan
    else:
        return np.nansum(a, **kwargs)
    
def copy2cache(src_path, cache_path, check=False):
    if os.path.isfile(cache_path):
        os.remove(cache_path)
    cmd = 'cp {:s} {:s}'.format(str(src_path), str(cache_path))
    return subprocess.run(cmd.split(' '), capture_output=False, check=check)

def copy2dist(cache_path, dist_path, check=False):
    if os.path.isfile(dist_path):
        os.remove(dist_path)
    cmd = 'cp {:s} {:s}'.format(str(cache_path), str(dist_path))
    return subprocess.run(cmd.split(' '), capture_output=False, check=check)

def build_fdir(fdir):
    fdir = str(fdir)
    if not os.path.exists(fdir):
        print('create directory: {:s}'.format(str(fdir)))
        os.makedirs(fdir, exist_ok=True)