from pathlib import Path, PosixPath
import os
import sys
import re
import inspect
import numpy as np
import json

DIR_INIT = True
OUT_JSON = True

""" project general settings """
try:
    HOME_DIR = Path.home()
except:
    HOME_DIR = Path(os.path.expanduser("~"))
PROJECT_DIR = Path(os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe()))))
PY_LIB_DIR = PROJECT_DIR / 'pylib'
CONFIG_DIR = PROJECT_DIR / 'configs'

sys.path.insert(0, str(PROJECT_DIR))
sys.path.insert(0, str(PY_LIB_DIR))

from utils import *
settings = DataContainer()

settings.HOME_DIR = HOME_DIR
settings.PROJECT_DIR = PROJECT_DIR
settings.PY_LIB_DIR = PY_LIB_DIR
settings.CONFIG_DIR = CONFIG_DIR

""" dataset: nature scene dataset """
NSD = DataContainer()
NSD.dir = DataContainer()
NSD.dir.root = settings.HOME_DIR / 'Data/Datasets/Natural-Scenes-Dataset'
NSD.dir.main = NSD.dir.root / 'nsddata'
NSD.dir.betas = NSD.dir.root / 'nsddata_betas'
NSD.dir.diffusion = NSD.dir.root / 'nsddata_diffusion'
NSD.dir.raw = NSD.dir.root / 'nsddata_rawdata'
NSD.dir.stimuli = NSD.dir.root / 'nsddata_stimuli'
NSD.dir.timeseries = NSD.dir.root / 'nsddata_timeseries'

NSD.session_all = ['subj{:02d}'.format(i+1) for i in range(8)]

settings.NSD = NSD

""" dataset: HCP 7T dataset """
HCP7T = DataContainer()
HCP7T.dir = DataContainer()
HCP7T.dir.root = settings.HOME_DIR / 'Data/Datasets/HCP-S1200-7T'
with open(settings.CONFIG_DIR / 'HCP7T_subject_IDs.txt') as f:
    HCP7T.subjID_all = [int(line.rstrip()) for line in f]
settings.HCP7T = HCP7T

""" dataset: HCP 3T dataset """
HCP3T = DataContainer()
HCP3T.dir = DataContainer()
HCP3T.dir.root = settings.HOME_DIR / 'Data/Datasets/HCP-S900-3T'
with open(settings.CONFIG_DIR / 'HCP3T_subject_IDs.txt') as f:
    HCP3T.subjID_all = [int(line.rstrip()) for line in f]
settings.HCP3T = HCP3T

""" dataset: HCP Aging dataset """
HCPAging = DataContainer()
HCPAging.dir = DataContainer()
HCPAging.dir.root = settings.HOME_DIR / 'Data/Datasets/HCP-Aging'
with open(settings.CONFIG_DIR / 'HCPAging_subject_IDs.txt') as f:
    HCPAging.subjID_all = [line.rstrip() for line in f]
settings.HCPAging = HCPAging

""" cache data (local) """
cacheData = DataContainer()
cacheData.dir = settings.HOME_DIR / 'Data/Cache/fMRI-State'
settings.cacheData = cacheData

""" project data """
projectData = DataContainer()
projectData.dir = DataContainer()
projectData.rel_dir = DataContainer()
projectData.dir.root = settings.HOME_DIR / 'Data/Project-Data/fMRI-States'
projectData.dir.general = projectData.dir.root / 'general'

projectData.dir.sessions_NSD = projectData.dir.root / 'sessions_NSD'
projectData.dir.general_NSD = projectData.dir.root / 'sessions_NSD' / 'general'
projectData.rel_dir.sessions_NSD = AttrDict(
    ROI_fMRISigs = 'fMRI_ROIs',
    Atlas = 'Atlas', # atals registered to subject space
    Nuisance = 'Nuisance',
)
projectData.rel_dir.general_NSD = AttrDict(
    Models = 'Models',
)

projectData.dir.sessions_HCP3T = projectData.dir.root / 'sessions_HCP3T'
projectData.dir.general_HCP3T = projectData.dir.root / 'sessions_HCP3T' / 'general'
projectData.rel_dir.sessions_HCP3T = AttrDict(
    ROI_fMRISigs = 'fMRI_ROIs',
    Nuisance = 'Nuisance',
)
projectData.rel_dir.general_HCP3T = AttrDict(
    Atlas = 'Atlas', # atals registered to MNI-2mm space
    Models = 'Models',
)
    
projectData.dir.sessions_HCP7T = projectData.dir.root / 'sessions_HCP7T'
projectData.dir.general_HCP7T = projectData.dir.root / 'sessions_HCP7T' / 'general'
projectData.rel_dir.sessions_HCP7T = AttrDict(
    ROI_fMRISigs = 'fMRI_ROIs',
    Nuisance = 'Nuisance'
)
projectData.rel_dir.general_HCP7T = AttrDict(
    Atlas = 'Atlas', # atals registered to MNI-1.6mm space
    Models = 'Models',
)

projectData.dir.sessions_HCPAging = projectData.dir.root / 'sessions_HCPAging'
projectData.dir.general_HCPAging = projectData.dir.root / 'sessions_HCPAging' / 'general'
projectData.rel_dir.sessions_HCPAging = AttrDict(
    ROI_fMRISigs = 'fMRI_ROIs',
    Nuisance = 'Nuisance',
)
projectData.rel_dir.general_HCPAging = AttrDict(
    Atlas = 'Atlas', # atals registered to MNI-2mm space
    Models = 'Models',
)

projectData.dir.sessions_PD = projectData.dir.root / 'sessions_PD'
projectData.dir.general_PD = projectData.dir.root / 'sessions_PD' / 'general'
projectData.rel_dir.sessions_PD = AttrDict(
    ROI_fMRISigs = 'fMRI_ROIs',
)
projectData.rel_dir.general_PD = AttrDict(
    Atlas = 'Atlas', # atals registered to MNI-2mm space
    Models = 'Models',
)

files = DataContainer()

files.general = AttrDict(
    AtlasInfo = 'atals_info.pkl',
)

files.sessions_NSD = AttrDict(
    SessTmplInfo = 'session_template_info.pkl',
    BrainMask_1mm = 'brainmask_1mm.nii.gz',
    BrainMask_1pt8mm = 'brainmask_1pt8mm.nii.gz',
    PGTempl_1mm = 'PG_volume_template_1mm.nii.gz',
    PGTempl_1pt8mm = 'PG_volume_template_1pt8mm.nii.gz',

    SessData = 'session_data.pkl',
    NuisanceInfo = 'nuisance_info.pkl',
    RoiFmriInfo = 'roi_fmri_info.pkl',
)
files.general_NSD = AttrDict(
    TrainTestData = 'train_test_data.pkl',
    PretrainedStateModels = 'pretrained_state_models.pkl',
    TransferredStateModels = 'transferred_state_models.pkl',
    BaselineModels = 'baseline_models.pkl',
    SessTaskScores = 'session_task_scores.pkl',
    StateImportance = 'transient_state_importance.pkl',
    TransientStates = 'transient_state_features.pkl',
)

files.sessions_HCP3T = AttrDict(
    RoiFmriInfo = 'roi_fmri_info.pkl',
    NuisanceInfo = 'nuisance_info.pkl',
    SessData = 'session_data.pkl',
)
files.general_HCP3T = AttrDict(
    BrainMask = 'brainmask_MNI_2mm.nii.gz',
    PGTemplate = 'PG_template_MNI_2mm.nii.gz',
    TemplateInfo = 'template_info.pkl',
    TrainTestData = 'train_test_data.pkl',
    BehaviorData = 'behaviors_data.pkl',
    CCAModeData = 'Canoncorr_FC_BEH.mat',

    PretrainedStateModels = 'pretrained_state_models.pkl',
    TransferredStateModels = 'transferred_state_models.pkl',
    BaselineModels = 'baseline_models.pkl'
)
    
files.sessions_HCP7T = AttrDict(
    RoiFmriInfo = 'roi_fmri_info.pkl',
    NuisanceInfo = 'nuisance_info.pkl',
    SessData = 'session_data.pkl',
)
files.general_HCP7T = AttrDict(
    BrainMask = 'brainmask_MNI_1pt6mm.nii.gz',
    PGTemplate = 'PG_template_MNI_1pt6mm.nii.gz',
    TemplateInfo = 'template_info.pkl',

    TrainTestData = 'train_test_data.pkl',
    PretrainedStateModels = 'pretrained_state_models.pkl',

    BehaviorData = 'behaviors_data.pkl',
    NeuroEvents = 'neural_events_data.pkl',
)

files.sessions_HCPAging = AttrDict(
    RoiFmriInfo = 'roi_fmri_info.pkl',
    SessData = 'session_data.pkl',
)
files.general_HCPAging = AttrDict(
    BrainMask = 'brainmask_MNI_2mm.nii.gz',
    PGTemplate = 'PG_template_MNI_2mm.nii.gz',
    TemplateInfo = 'template_info.pkl',
    subjectInfo = 'HCP-Aging_subject_info.csv',

    BehaviorData = 'behaviors_data.pkl',
    TrainTestData = 'train_test_data.pkl',
    PretrainedStateModels = 'pretrained_state_models.pkl',
    TransferredStateModels = 'transferred_state_models.pkl',
    BaselineModels = 'baseline_models.pkl'
)

files.sessions_PD = AttrDict(
    RoiFmriInfo = 'roi_fmri_info.pkl',
    SessData = 'session_data.pkl',
)
files.general_PD = AttrDict(
    BrainMask = 'brainmask_MNI_2mm.nii.gz',
    PGTemplate = 'PG_template_MNI_2mm.nii.gz',
    TemplateInfo = 'template_info.pkl',
    sessionInfo = 'PD_sessions_info.csv',

    TrainTestData = 'train_test_data.pkl',
    PretrainedStateModels = 'pretrained_state_models.pkl',
    TransferredStateModels = 'transferred_state_models.pkl',
    BaselineModels = 'baseline_models.pkl'
)

projectData.files = files
settings.projectData = projectData

""" project results """
results = DataContainer()
results.dir = settings.PROJECT_DIR / 'results'

settings.results = results


def build_fdir(fdir):
    fdir = str(fdir)
    if not os.path.exists(fdir):
        print('create directory: {:s}'.format(str(fdir)))
        os.makedirs(fdir, exist_ok=True)

""" Initialize project data folder """
if DIR_INIT:
    print('initializing ... ')
    for fdir in [projectData.dir.sessions_NSD, 
                 projectData.dir.general_NSD,
                 projectData.dir.sessions_HCP3T, 
                 projectData.dir.general_HCP3T,
                 projectData.dir.sessions_HCP7T, 
                 projectData.dir.general_HCP7T,
                 projectData.dir.sessions_HCPAging, 
                 projectData.dir.general_HCPAging,
                 projectData.dir.sessions_PD, 
                 projectData.dir.general_PD,
                 projectData.dir.general]:
        build_fdir(fdir)
    
    for key, rel_dir in projectData.rel_dir.general_HCP3T.items():
        build_fdir(projectData.dir.general_HCP3T / rel_dir)

    for key, rel_dir in projectData.rel_dir.general_HCP7T.items():
        build_fdir(projectData.dir.general_HCP7T / rel_dir)

    for key, rel_dir in projectData.rel_dir.general_HCPAging.items():
        build_fdir(projectData.dir.general_HCPAging / rel_dir)

    for key, rel_dir in projectData.rel_dir.general_NSD.items():
        build_fdir(projectData.dir.general_NSD / rel_dir)

    for key, rel_dir in projectData.rel_dir.general_PD.items():
        build_fdir(projectData.dir.general_PD / rel_dir)

    for sess_id in settings.NSD.session_all:
        sess_dir = projectData.dir.sessions_NSD / '{:s}'.format(sess_id)
        build_fdir(sess_dir)
        for key, rel_dir in projectData.rel_dir.sessions_NSD.items():
            build_fdir(sess_dir / rel_dir)
    
    for sess_id in settings.HCP7T.subjID_all:
        sess_dir = projectData.dir.sessions_HCP7T / '{:d}'.format(sess_id)
        build_fdir(sess_dir)
        for key, rel_dir in projectData.rel_dir.sessions_HCP7T.items():
            build_fdir(sess_dir / rel_dir)

    for sess_id in settings.HCP3T.subjID_all:
        sess_dir = projectData.dir.sessions_HCP3T / '{:d}'.format(sess_id)
        build_fdir(sess_dir)
        for key, rel_dir in projectData.rel_dir.sessions_HCP3T.items():
            build_fdir(sess_dir / rel_dir)

    for sess_id in settings.HCPAging.subjID_all:
        sess_dir = projectData.dir.sessions_HCPAging / '{:s}'.format(sess_id)
        build_fdir(sess_dir)
        for key, rel_dir in projectData.rel_dir.sessions_HCPAging.items():
            build_fdir(sess_dir / rel_dir)


""" Generating JSON file for settings """
def serialize(obj):
    if type(obj) is PosixPath:
        return str(obj)
    
    if type(obj) is DataContainer:
        return obj.__dict__

if OUT_JSON:
    print('generating json ...')
    with open(str(settings.PROJECT_DIR / "settings.json"), "w") as outfile:
        json_string = json.dumps(settings, default=serialize, indent=4)
        outfile.write(json_string)

