
import sys, pandas as pd, matplotlib.pyplot as plt, matplotlib as mpl
from cycler import cycler

def load_csv(p): return pd.read_csv(p)

def annotate_points(ax, pts_df, metric):
    if pts_df is None: return
    for _,r in pts_df.iterrows():
        ax.scatter([r["coverage"]],[r[metric]], zorder=4, s=25, facecolors='none', edgecolors='k')
        ax.annotate(r["label"], (r["coverage"], r[metric]),
                    xytext=(6,6), textcoords="offset points", fontsize=7,
                    bbox=dict(boxstyle="round,pad=0.2", fc="white", ec="0.8", lw=0.5))

def plot_one(base_csv, prg_csv, out_pdf, metric, pts_csv=None,
             base_label="Baseline", prg_label="PragAURA (per-act)"):
    mpl.style.use('scripts/style/iclr.mplstyle')
    mpl.rcParams['axes.prop_cycle'] = cycler(color=['#0072B2','#D55E00','#009E73','#CC79A7'])
    mpl.rcParams['axes.prop_cycle'] = cycler(color=['#0072B2','#D55E00','#009E73','#CC79A7'])
    db, dp = load_csv(base_csv), load_csv(prg_csv)
    pts = pd.read_csv(pts_csv) if pts_csv else None

    fig = plt.figure(figsize=(3.3, 3.0))  # good single-column size
    ax = fig.add_subplot(111)

    ax.plot(db["coverage"], db[metric], marker="o", linestyle="--", label=base_label)
    ax.plot(dp["coverage"], dp[metric], marker="o", linestyle="-",  label=prg_label)

    annotate_points(ax, pts, metric)

    ax.set_xlabel("Coverage (1 – abstain)")
    ax.set_ylabel(metric)
    ax.set_xlim(0.0, 1.0); ax.set_ylim(0.0, 1.0)
    ax.legend(frameon=False, loc="lower left")
    fig.tight_layout()
    fig.savefig(out_pdf, bbox_inches="tight")
    print("Wrote", out_pdf)

if __name__=="__main__":
    if len(sys.argv) < 5:
        print("Usage: plot_rc_iclr.py base.csv prg.csv out.pdf METRIC [points.csv]")
        sys.exit(1)
    base, prg, out, metric = sys.argv[1:5]
    pts = sys.argv[5] if len(sys.argv)>5 else None
    plot_one(base, prg, out, metric, pts)
