#    Copyright 2024 Hao Zhang
#
#    Licensed under the Apache License, Version 2.0 (the "License");
#    you may not use this file except in compliance with the License.
#    You may obtain a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS,
#    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#    See the License for the specific language governing permissions and
#    limitations under the License.


from typing import List, Optional, Tuple, Union, Dict
import torch
import torch.nn as nn

from transformers import AutoConfig, AutoModelForCausalLM

from transformers.modeling_outputs import CausalLMOutputWithPast
from transformers.generation.utils import GenerateOutput

from ..llava_arch import LlavaMetaModel, LlavaMetaForCausalLM, RLlavaMetaForCausalLM
from transformers import Qwen2Config, Qwen2Model, Qwen2ForCausalLM



class LlavaQwen2Config(Qwen2Config):
    model_type = "llava_qwen2"

    def __init__(self, shuffle_patches: bool = False, log_debug: bool = False, **kwargs):
        self.shuffle_patches = shuffle_patches
        self.log_debug = log_debug
        super(LlavaQwen2Config, self).__init__(**kwargs)

class RLlavaQwen2Config(Qwen2Config):
    model_type = "Rllava_qwen2"

    def __init__(
        self,
        region_sort: str = "default",
        region_source: str = "passed",
        region_cluster_args: str = "t=0.7,m=2",
        region_interpolate: str = "upsample",
        region_late_pe: bool = False,
        region_pooling_method: str = "average", # avg, max, or cross-attn
        region_attn_args: dict = None,
        region_extra: str = "none",
        region_filter: str = "none",
        region_expand_mult: int = 1,
        log_debug: bool = False,
        **kwargs
    ):
        self.region_sort = region_sort
        self.region_source = region_source
        self.region_cluster_args = region_cluster_args
        self.region_late_pe = region_late_pe
        self.region_interpolate = region_interpolate
        self.region_pooling_method = region_pooling_method
        self.region_attn_args = region_attn_args
        self.region_extra = region_extra
        self.region_filter = region_filter
        self.region_expand_mult = region_expand_mult
        self.log_debug = log_debug
        super(RLlavaQwen2Config, self).__init__(**kwargs)



class LlavaQwen2Model(LlavaMetaModel, Qwen2Model):
    config_class = LlavaQwen2Config

    def __init__(self, config: Qwen2Config):
        super(LlavaQwen2Model, self).__init__(config)


class LlavaQwen2ForCausalLM(Qwen2ForCausalLM, LlavaMetaForCausalLM):
    config_class = LlavaQwen2Config

    def __init__(self, config):
        # super(Qwen2ForCausalLM, self).__init__(config)
        Qwen2ForCausalLM.__init__(self, config)
        config.model_type = "llava_qwen2"
        config.rope_scaling = None
        self.shuffle_patches = config.shuffle_patches

        self.model = LlavaQwen2Model(config)
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)
        if config.log_debug:
            self.debug = True
            self.logs={
                "loss": [],
                "logits": torch.zeros(config.vocab_size, dtype=torch.float32, requires_grad=False)
            }
        else:
            self.debug = False

        # Initialize weights and apply final processing
        self.post_init()

    def get_model(self):
        return self.model

    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        images: Optional[torch.FloatTensor] = None,
        image_sizes: Optional[List[List[int]]] = None,
        return_dict: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
    ) -> Union[Tuple, CausalLMOutputWithPast]:

        if inputs_embeds is None:
            (input_ids, position_ids, attention_mask, past_key_values, inputs_embeds, labels) = self.prepare_inputs_labels_for_multimodal(input_ids, position_ids, attention_mask, past_key_values, labels, images, image_sizes)

        output = super().forward(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            labels=labels,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            cache_position=cache_position
        )
        if self.debug:
            self.logs["logits"] += output.logits[labels >= 0].mean(dim=0).cpu().detach()
            self.logs["loss"].append(output.loss.item())
        return output

    @torch.no_grad()
    def generate(
        self,
        inputs: Optional[torch.Tensor] = None,
        images: Optional[torch.Tensor] = None,
        image_sizes: Optional[torch.Tensor] = None,
        **kwargs,
    ) -> Union[GenerateOutput, torch.LongTensor]:
        position_ids = kwargs.pop("position_ids", None)
        attention_mask = kwargs.pop("attention_mask", None)
        if "inputs_embeds" in kwargs:
            raise NotImplementedError("`inputs_embeds` is not supported")

        if images is not None:
            (
                inputs,
                position_ids,
                attention_mask,
                _,
                inputs_embeds,
                _
            ) = self.prepare_inputs_labels_for_multimodal(
                inputs,
                position_ids,
                attention_mask,
                None,
                None,
                images,
                image_sizes=image_sizes
            )
        else:
            inputs_embeds = self.get_model().embed_tokens(inputs)

        return super().generate(
            position_ids=position_ids,
            attention_mask=attention_mask,
            inputs_embeds=inputs_embeds,
            **kwargs
        )

    def prepare_inputs_for_generation(self, input_ids, past_key_values=None, inputs_embeds=None, **kwargs):
        images = kwargs.pop("images", None)
        image_sizes = kwargs.pop("image_sizes", None)
        inputs = super().prepare_inputs_for_generation(input_ids, past_key_values=past_key_values, inputs_embeds=inputs_embeds, **kwargs)
        if images is not None:
            inputs["images"] = images
        if image_sizes is not None:
            inputs["image_sizes"] = image_sizes
        return inputs

import time
class RLlavaQwen2ForCausalLM(Qwen2ForCausalLM, RLlavaMetaForCausalLM):
    config_class = RLlavaQwen2Config

    def __init__(self, config: RLlavaQwen2Config):
        # super(Qwen2ForCausalLM, self).__init__(config)
        Qwen2ForCausalLM.__init__(self, config)
        config.model_type = "Rllava_qwen2"
        config.rope_scaling = None
        self.region_sort = config.region_sort
        self.region_source = config.region_source
        self.region_cluster_args = config.region_cluster_args
        self.region_late_pe = config.region_late_pe
        self.region_interpolate = config.region_interpolate
        self.region_pooling_method = config.region_pooling_method
        self.region_attn_args = config.region_attn_args
        self.region_extra = config.region_extra
        self.region_filter = config.region_filter
        self.region_expand_mult = config.region_expand_mult
        self.model = LlavaQwen2Model(config)
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)
        if config.log_debug:
            self.debug = True
            self.logs={
                "loss": [],
                "regions": [],
                "idx": [],
                "logits": torch.zeros(config.vocab_size, dtype=torch.float32, requires_grad=False)
            }
        else:
            self.debug = False

        # Initialize weights and apply final processing
        self.post_init()

    @property
    def need_region_upproj(self):
        return self.region_expand_mult!= 1 and self.region_attn_args in ["average", "max"]

    def get_model(self):
        return self.model

    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[List[torch.FloatTensor]] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        labels: Optional[torch.LongTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        images: Optional[torch.FloatTensor] = None,
        sam_masks: Optional[List[torch.BoolTensor]] = None,
        image_sizes: Optional[List[List[int]]] = None,
        return_dict: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
        idx: Optional[torch.Tensor] = None,
        return_masks: Optional[bool] = False,
    ) -> Union[Tuple, CausalLMOutputWithPast]:
        if inputs_embeds is None:
            (
                input_ids,
                position_ids,
                attention_mask,
                past_key_values,
                inputs_embeds,
                labels
            ) = self.prepare_inputs_labels_for_multimodal(
                input_ids,
                position_ids,
                attention_mask,
                past_key_values,
                labels,
                images,
                sam_masks,
                image_sizes,
                return_masks=return_masks
            )

        output = super().forward(
            input_ids=input_ids,
            attention_mask=attention_mask,
            position_ids=position_ids,
            past_key_values=past_key_values,
            inputs_embeds=inputs_embeds,
            labels=labels,
            use_cache=use_cache,
            output_attentions=output_attentions,
            output_hidden_states=output_hidden_states,
            return_dict=return_dict,
            cache_position=cache_position
        )
        if self.debug:
            self.logs["logits"] += output.logits[labels >= 0].mean(dim=0).cpu().detach()
            self.logs["idx"].append(idx.item())
            self.logs["loss"].append(output.loss.item())
            self.logs["regions"].append([len(s) for s in sam_masks])
        return output

    @torch.no_grad()
    def generate(
        self,
        inputs: Optional[torch.Tensor] = None,
        images: Optional[torch.Tensor] = None,
        sam_masks: Optional[List[torch.BoolTensor]] = None,
        image_sizes: Optional[torch.Tensor] = None,
        **kwargs,
    ) -> Union[GenerateOutput, torch.LongTensor]:
        position_ids = kwargs.pop("position_ids", None)
        attention_mask = kwargs.pop("attention_mask", None)
        if "inputs_embeds" in kwargs:
            raise NotImplementedError("`inputs_embeds` is not supported")

        if images is not None:
            (
                inputs,
                position_ids,
                attention_mask,
                _,
                inputs_embeds,
                _
            ) = self.prepare_inputs_labels_for_multimodal(
                inputs,
                position_ids,
                attention_mask,
                None,
                None,
                images,
                sam_masks,
                image_sizes=image_sizes
            )
        else:
            inputs_embeds = self.get_model().embed_tokens(inputs)

        return super().generate(
            position_ids=position_ids,
            attention_mask=attention_mask,
            inputs_embeds=inputs_embeds,
            **kwargs
        )

    def prepare_inputs_for_generation(self, input_ids, past_key_values=None,
                                      inputs_embeds=None, **kwargs):
        images = kwargs.pop("images", None)
        sam_masks = kwargs.pop("sam_masks", None)
        image_sizes = kwargs.pop("image_sizes", None)
        inputs = super().prepare_inputs_for_generation(
            input_ids, past_key_values=past_key_values, inputs_embeds=inputs_embeds, **kwargs
        )
        if images is not None:
            inputs['images'] = images
        if sam_masks is not None:
            inputs['sam_masks'] = sam_masks
        if image_sizes is not None:
            inputs['image_sizes'] = image_sizes
        return inputs

AutoConfig.register("llava_qwen2", LlavaQwen2Config)
AutoConfig.register("Rllava_qwen2", RLlavaQwen2Config)
AutoModelForCausalLM.register(LlavaQwen2Config, LlavaQwen2ForCausalLM)
AutoModelForCausalLM.register(RLlavaQwen2Config, RLlavaQwen2ForCausalLM)
