import os
import json
import csv
from collections import OrderedDict
import argparse

def flatten_dict(d, parent_key='', sep='.'):
    """
    Recursively flattens a nested dict. 
    Nested keys are joined by sep.
    """
    items = {}
    for k, v in d.items():
        # new_key = f"{parent_key}{sep}{k}" if parent_key else k
        new_key = k
        if isinstance(v, dict):
            items.update(flatten_dict(v, new_key, sep=sep))
        else:
            items[new_key] = v
    return items

def summarize_jsons_to_csv(root_dir, output_filename='result.csv'):
    rows = []
    all_columns = []

    # 1) Walk directory tree
    for dirpath, _, filenames in os.walk(root_dir):
        for fname in filenames:
            if fname.lower().endswith('.json'):
                rel_dir = os.path.relpath(dirpath, root_dir)
                rel_path = os.path.join(rel_dir, fname) if rel_dir != '.' else fname
                file_key = rel_path.replace('.json', '')
                json_path = os.path.join(dirpath, fname)

                with open(json_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                if not isinstance(data, dict):
                    print(f"Warning: skipping {json_path}: top-level is not a dict")
                    continue

                flat = flatten_dict(data)
                flat['row_name'] = file_key
                rows.append(flat)
                for key in flat.keys():
                    if key not in all_columns and key != 'row_name':
                        all_columns.append(key)

    rows.sort(key=lambda r: r['row_name'])
    all_columns = [c for c in all_columns if '(' not in c] + [c for c in all_columns if '(' in c]
    columns = ['row_name'] + all_columns

    # 4) Write CSV
    output_path = output_filename
    os.makedirs(os.path.dirname(output_path), exist_ok=True)
    with open(output_path, 'w', newline='', encoding='utf-8') as csvfile:
        writer = csv.DictWriter(csvfile, fieldnames=columns)
        writer.writeheader()
        for row in rows:
            row_out = {col: row.get(col, '') for col in columns}
            writer.writerow(row_out)

    print(f"Summarized {len(rows)} JSON files into {output_path}")

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--root_dir', type=str)
    parser.add_argument('--output', type=str)
    args = parser.parse_args()

    summarize_jsons_to_csv(args.root_dir, args.output)
