import kconfiglib
import json

# load the top-level Kconfig file
kconfig = kconfiglib.Kconfig("Kconfig")

nodes = []
edges = []


visited = set()  # add a set to store visited nodes
edge_set = set()

def get_node_id(node):
    """Unified function to get node ID"""
    if isinstance(node.item, kconfiglib.Symbol):
        return id(node.item)
    return id(node)

visited_nodes = set()  # global variable to store visited nodes

def extract_graph(node):
    """recursion function to extract the graph"""
    while node:
        current_id = get_node_id(node)
        
        if current_id in visited_nodes:
            node = node.next
            continue
            
        visited_nodes.add(current_id)
        
        # process the node
        if node.item == kconfiglib.MENU:
            nodes.append({
                "id": current_id,
                "type": "menu",
                "name": node.prompt[0],
                #"visibility": str(node.item.visibility) if hasattr(node.item, 'visibility') else None
            })
        elif isinstance(node.item, kconfiglib.Symbol):
            select_info = []
            for select, cond in node.item.selects:
                select_info.append({
                    "selected": select.name,  # selected configuration name
                    "condition": str(cond) if cond else None  # select condition
                })                
            
            
            # Get the readable name of the configuration type
            type_map = {
                kconfiglib.BOOL: "bool",
                kconfiglib.TRISTATE: "tristate",
                kconfiglib.INT: "int",
                kconfiglib.HEX: "hex",
                kconfiglib.STRING: "string"
            }
            config_type = type_map.get(node.item.orig_type, str(node.item.orig_type))
            
            
            nodes.append({
                "id": current_id,
                "type": "config",
                "name": node.item.name,
                "config_type": config_type,
                "def_value": node.item.str_value,
                "ranges": [(str(low), str(high), str(cond)) for low, high, cond in node.item.ranges],
                #"def_val_cond": [{"value": str(expr), "condition": str(cond) if cond else None} for expr, cond in node.item.defaults],
                "defaults": [(str(value), str(cond)) for value, cond in node.item.defaults],  
                "select": select_info,
                "depends": str(node.dep) if node.dep else None,
                #"visibility": str(node.item.visibility) if hasattr(node.item, 'visibility') else None,
                "help": str(node.help) if node.help else None
            })
        elif isinstance(node.item, kconfiglib.Choice):
            # process the choice options
            choice_options = []
            for sym in node.item.syms:
                option_info = {
                    "name": sym.name,
                    "def_value": sym.str_value,
                    "visibility": str(sym.visibility) if hasattr(sym, 'visibility') else None
                }
                choice_options.append(option_info)
                

            type_map = {
                kconfiglib.BOOL: "bool",
                kconfiglib.TRISTATE: "tristate",
                kconfiglib.INT: "int",
                kconfiglib.HEX: "hex",
                kconfiglib.STRING: "string"
            }
            choice_type = type_map.get(node.item.orig_type, str(node.item.orig_type))
            
            nodes.append({
                "id": current_id,
                "type": "choice",
                "name": node.prompt[0],
                "choice_type": choice_type,  
                "optional": node.item.orig_type == kconfiglib.TRISTATE, 
                "depends": str(node.dep) if node.dep else None,
                #"visibility": str(node.item.visibility) if hasattr(node.item, 'visibility') else None,
                "opt_info": choice_options,
                "help": str(node.help) if node.help else None  
            })

        # Only process parent-child relationship edges
        if node.list:
            child = node.list
            while child:
                child_id = get_node_id(child)
                # Check the types of parent nodes and child nodes
                parent_type = "menu"
                if isinstance(node.item, kconfiglib.Symbol):
                    parent_type = "config"
                elif isinstance(node.item, kconfiglib.Choice):
                    parent_type = "choice"
                
                child_type = "menu"
                if isinstance(child.item, kconfiglib.Symbol):
                    child_type = "config"
                elif isinstance(child.item, kconfiglib.Choice):
                    child_type = "choice"
                
                edge_key = (current_id, child_id, "parent")
                if edge_key not in edge_set:
                    edge_set.add(edge_key)
                    edges.append({
                        "source": current_id,
                        "target": child_id,
                        "type": "parent",
                        "source_type": parent_type,
                        "target_type": child_type
                    })
                
                if child_id not in visited_nodes:
                    extract_graph(child)
                child = child.next

        node = node.next
        
# Usage
visited_nodes.clear()  # clear the visited nodes
extract_graph(kconfig.top_node)

print(len(nodes), len(edges))
# store nodes and edges to a JSON file
output_data = {"nodes": nodes, "edges": edges}

with open('/data/y_book/project/AutoOS_rag/test/kconfig_graph.json', 'w') as json_file:
    json.dump(output_data, json_file, indent=4)

print("Graph data has been written to kconfig_graph.json")