class TableVisualizer:
    """
    Render a simple and functional HTML table. Typically used to visualize results.
    table_configs is a list of configs, one element for each column of the tabular data, each of which shoudl contain:
    'id' : Id of the column
    'display_name' : Display name of the column (if different from the 'id')
    'sortable' (bool) : If the column elements require a sort functionality
    'type' (text / image / hyperlink) (str) : This determines the display for the particular column
    'height' : Height on the image (only valid if type is set to image)
    """

    def __init__(self, table_configs, css_style_path, out_file_name):
        self.table_configs = table_configs
        self.validate_configs()
        self.num_columns = len(self.table_configs)
        self.out_file_name = out_file_name
        self.css_style_path = css_style_path
        self.data = []

    def validate_configs(self):
        for column in self.table_configs:
            if "id" not in column:
                raise ValueError(
                    "Field id not present for some elements in the table_configs"
                )
            if "display_name" not in column:
                column["display_name"] = column["id"]
            if "sortable" not in column:
                column["sortable"] = False
            if "type" not in column:
                raise ValueError(
                    "Field type not present for some elements in the table_configs"
                )
            if "height" not in column:
                column["height"] = -1
            if "hidden" not in column:
                column["hidden"] = False

    def add_row(self, row_data):
        if len(row_data) != self.num_columns:
            raise ValueError("Incorrect number of entries in the row data.")
        self.data.append(row_data)
        # print("len of cur data", len(self.data))

    def render(self):
        # print("configs", self.table_configs)

        with open(self.out_file_name, "w") as f:
            f.write("<html>\n")

            f.write('<div id="viztable">\n')
            self.add_search_sort_functionality(f)

            f.write("<table>\n")
            self.add_table_head(f)
            self.add_table_body(f)
            f.write("</table>\n")

            f.write("</div>\n")
            self.add_html_head(f)
            f.write("</html>")

    def add_search_sort_functionality(self, f):
        f.write('<input class="search" placeholder="Search" />\n<br>\n')
        for column in self.table_configs:
            if column["sortable"]:
                f.write('<button class="sort" data-sort="{0}">'.format(column["id"]))
                f.write("Sort by {0}".format(column["display_name"]))
                f.write("</button>\n")
        f.write("<p>\n")

    def add_table_head(self, f):
        f.write("<thead>\n")
        f.write("<tr>\n")
        for column in self.table_configs:
            hidden_str = ' style="display:none;""'
            if not column["hidden"]:
                hidden_str = ""
            f.write(
                '<td {0} width={1} class="{2}">{3}</td>\n'.format(
                    hidden_str, column["width"], column["id"], column["display_name"]
                )
            )
        f.write("</tr>\n")
        f.write("</thead>\n")

    def add_table_body(self, f):
        f.write('<tbody class="list">\n')
        # print("configs", self.table_configs)

        for current_data in self.data:
            # print("current data", current_data)

            f.write("<tr>\n")
            for col_count, col_data in enumerate(current_data):
                hidden_str = 'style="display:none;'
                # print("reached")
                if not self.table_configs[col_count]["hidden"]:
                    hidden_str = ""

                f.write(
                    '<td {0} class="{1}">\n'.format(
                        hidden_str, self.table_configs[col_count]["id"]
                    )
                )
                # print("wrote first col")
                if self.table_configs[col_count]["type"] == "text":
                    f.write("{}".format(col_data))
                    # print("wrote text")
                elif self.table_configs[col_count]["type"] == "image":
                    f.write(
                        '<img src="{}" height={}>'.format(
                            col_data, self.table_configs[col_count]["height"]
                        )
                    )
                    # print("wrote image")
                elif self.table_configs[col_count]["type"] == "hyperlink":
                    f.write(
                        '<a href="{}" target=“_blank”>{}</a>'.format(
                            col_data[0], col_data[1]
                        )
                    )
                # print("reached end")
                f.write("</td>\n")
            f.write("</tr>\n")

        f.write("</tbody>\n")

    def add_html_head(self, f):
        f.write("<head>\n")
        f.write("<title>Data visualization</title>\n")
        f.write('<script src="list.min.js"></script>\n')
        self.add_js(f)
        self.add_style(f)
        f.write("</head>")

    def add_js(self, f):
        column_names_repr = []
        for column in self.table_configs:
            column_names_repr.append('"{}"'.format(column["id"]))
        column_names_repr = ",".join(column_names_repr)

        f.write("<script>\n")
        f.write("var options = {{valueNames: [ {0} ]}};\n".format(column_names_repr))
        f.write('var gameList = new List("viztable", options);\n')
        f.write("</script>\n")

    def add_style(self, f):
        with open(self.css_style_path) as fin:
            style_info = fin.readlines()
        f.write("<style>\n")
        f.writelines(style_info)
        f.write("</style>\n")