import copy
import hashlib
import matplotlib.pyplot as plt
import networkx as nx
import numpy as np
import os
import pickle
import pulp

from abc import ABC, abstractmethod
from matplotlib.colors import to_rgb
from matplotlib.ticker import FormatStrFormatter
from multiprocessing import Pool
from scipy.special import lambertw
from tqdm import tqdm
from typing import Callable

"""
Returns the smallest value in [0,1] that satisfies the condition function
"""
def binary_search_smallest_true(
        condition_fn: Callable[[float], bool],
        low: float = 0.0,
        high: float = 1.0,
        eps: float = 1e-6
    ) -> float:
    while (high - low) > eps:
        mid = (low + high) / 2.0
        if condition_fn(mid):
            high = mid
        else:
            low = mid
    return high

"""
Returns the largest value in [0,1] that satisfies the condition function
"""
def binary_search_largest_true(
        condition_fn: Callable[[float], bool],
        low: float = 0.0,
        high: float = 1.0,
        eps: float = 1e-6
    ) -> float:
    while (high - low) > eps:
        mid = (low + high) / 2.0
        if condition_fn(mid):
            low = mid
        else:
            high = mid
    return low

"""
Compute SHA-256 on string x then return truncated hex digest
"""
def short_hash_from_string(x, length=64):
    return hashlib.sha256(x.encode("utf-8")).hexdigest()[:length]

"""
Parses .mtx input to networkx graph
"""
def parse_mtx_to_nx(filename: str) -> nx.Graph:
    G = nx.Graph()
    n = None
    with open(filename, 'r') as file:
        for line in file:
            if line[0] != '%':
                lst = line.split()
                if n is None:
                    n = int(lst[0])
                else:
                    if len(lst) == 3:
                        G.add_edge(int(lst[0]), int(lst[1]), weight=float(lst[2]))
                    else:
                        G.add_edge(int(lst[0]), int(lst[1]))
    return G

"""
Parses .edge input to networkx graph
"""
def parse_edges_to_nx(filename: str) -> nx.Graph:
    G = nx.Graph()
    with open(filename, 'r') as file:
        for line in file:
            u_str, v_str, w_str = line.split()
            G.add_edge(int(u_str), int(v_str), weight=float(w_str))
    return G

"""
Processes graph to become a bipartite graph, just like https://arxiv.org/abs/1808.04863
1) Shuffle node indices
2) Take first n/2 as offline
3) Take next n/2 as online
4) Ignore all non-bipartite crossing edges
"""
def process_graph_to_bipartite(rng: np.random.Generator, G: nx.Graph) -> nx.Graph:
    G_new = nx.Graph()
    nodes = list(G.nodes())
    rng.shuffle(nodes)
    n = len(nodes) - (len(nodes) % 2)
    G_new.add_nodes_from([f"u{i}" for i in range(n // 2)], bipartite=0)
    G_new.add_nodes_from([f"v{j - (n // 2)}" for j in range(n // 2, n)], bipartite=1)
    for u, v, data in G.edges(data=True):
        i, j = min(u,v), max(u,v)
        if 0 <= i and i < n // 2 and n // 2 <= j and j < n:
            G_new.add_edge(f"u{i}", f"v{j - (n // 2)}", **data)
    assert nx.bipartite.is_bipartite(G_new)
    return G_new
