import torch
from torch.optim import Adam

from distributionalrl.model import Ensemble, QRDQN
from distributionalrl.utils import calculate_quantile_huber_loss, disable_gradients, evaluate_quantile_at_action, update_params

from .base_agent import BaseAgent


class QRDQNAgent(BaseAgent):

    def __init__(self, env, test_env, log_dir, frame_stack=4, num_steps=5*(10**7), algorithm="TD",
                 batch_size=32, N=200, kappa=1.0, lr=5e-5, memory_size=10**6,
                 gamma=0.99, multi_step=1, update_interval=4,
                 target_update_interval=10000, start_steps=50000,
                 epsilon_train=0.01, epsilon_eval=0.001,
                 epsilon_decay_steps=250000, double_q_learning=False, decision_function="TD",
                 dueling_net=False, noisy_net=False, alea_comparison_measure="wang", alea_comparison_hyps=(0.0),
                 epi_comparison_measure="oneversusall", epi_comparison_hyps=(0.0), epistemic_method="none", num_ensembles=1, 
                 use_per=False, log_interval=100, eval_interval=250000, num_eval_steps=125000,
                 max_episode_steps=27000, grad_cliping=None, cuda=True,
                 seed=0):
        
        log_dir = log_dir+'method='+epistemic_method +'-num_ensembles='+str(num_ensembles)+'-'+'epi='+epi_comparison_measure+'_'+'_'.join(map(str, epi_comparison_hyps))+'_'+'-alea='+alea_comparison_measure+'_'+'_'.join(map(str, alea_comparison_hyps))
        
        super(QRDQNAgent, self).__init__(
            env, test_env, log_dir, frame_stack, num_steps, algorithm, batch_size, memory_size,
            gamma, multi_step, update_interval, target_update_interval,
            start_steps, epsilon_train, epsilon_eval, epsilon_decay_steps,
            double_q_learning, decision_function, dueling_net, noisy_net, epistemic_method, use_per, log_interval,
            eval_interval, num_eval_steps, max_episode_steps, grad_cliping,
            cuda, seed)

        ensemble_comparison_measure = epi_comparison_measure if (epistemic_method == "ensemble") else "wang"
        ensemble_comparison_hyps = epi_comparison_hyps if (epistemic_method == "ensemble") else (0.0)
        self.num_ensembles = num_ensembles if (epistemic_method == "ensemble") else 1
        # Online network.
        self.online_net = Ensemble(QRDQN, self.device, epi_comparison_measure=ensemble_comparison_measure, epi_comparison_hyps=ensemble_comparison_hyps, num_ensembles=self.num_ensembles,
            state_shape=env.observation_space.shape,
            num_actions=self.num_actions, N=N, dueling_net=dueling_net,
            noisy_net=noisy_net)
        # Target network.
        self.target_net = Ensemble(QRDQN, self.device, epi_comparison_measure=ensemble_comparison_measure, epi_comparison_hyps=ensemble_comparison_hyps, num_ensembles=self.num_ensembles,
            state_shape=env.observation_space.shape,
            num_actions=self.num_actions, N=N, dueling_net=dueling_net,
            noisy_net=noisy_net)

        # Copy parameters of the learning network to the target network.
        self.update_target()
        # Disable calculations of gradients of the target network.
        disable_gradients(self.target_net.ensemble)

        self.optim = Adam(
            self.online_net.params(),
            lr=lr, eps=1e-2/batch_size)

        # Fixed fractions.
        taus = torch.arange(
            0, N+1, device=self.device, dtype=torch.float32) / N
        self.tau_hats = ((taus[1:] + taus[:-1]) / 2.0).view(1, N)

        self.N = N
        self.kappa = kappa

    def learn(self):
        self.learning_steps += 1
        self.online_net.sample_noise()
        self.target_net.sample_noise()

        if self.use_per:
            (states, actions, rewards, next_states, dones), weights =\
                self.memory.sample(self.batch_size*self.num_ensembles)
        else:
            states, actions, rewards, next_states, dones =\
                self.memory.sample(self.batch_size*self.num_ensembles)
            weights = None

        total_quantile_loss, total_errors = 0.0, 0.0
        for i in range(self.num_ensembles):
            batch_weights = None if weights is None else weights[i*self.batch_size:(i+1)*self.batch_size]

            quantile_loss, mean_q, errors = self.calculate_loss(
                states[i*self.batch_size:(i+1)*self.batch_size], actions[i*self.batch_size:(i+1)*self.batch_size], rewards[i*self.batch_size:(i+1)*self.batch_size], next_states[i*self.batch_size:(i+1)*self.batch_size], dones[i*self.batch_size:(i+1)*self.batch_size], batch_weights, i)
            assert errors.shape == (self.batch_size, 1)

            total_quantile_loss += quantile_loss
            total_errors += errors

        update_params(
            self.optim, quantile_loss,
            networks=[self.online_net],
            retain_graph=False, grad_cliping=self.grad_cliping)

        if self.use_per:
            self.memory.update_priority(errors)

        if self.episodes % self.log_interval == 0:
            self.writer.add_scalar(
                'loss/quantile_loss', quantile_loss.detach().item(),
                4*self.steps)
            self.writer.add_scalar('stats/mean_Q', mean_q, 4*self.steps)

    def calculate_loss(self, states, actions, rewards, next_states, dones,
                       weights, i_ens):

        # Calculate quantile values of current states and actions at taus.
        current_sa_quantiles = evaluate_quantile_at_action(
            self.online_net.ensemble[i_ens](states=states),
            actions)
        assert current_sa_quantiles.shape == (self.batch_size, self.N, 1)

        with torch.no_grad():
            # Calculate Q values of next states.
            if self.double_q_learning:
                # Sample the noise of online network to decorrelate between
                # the action selection and the quantile calculation.
                self.online_net.ensemble[i_ens].sample_noise()
                next_q = self.online_net.ensemble[i_ens].calculate_q(states=next_states)
            else:
                next_q = self.target_net.ensemble[i_ens].calculate_q(states=next_states)

            # Calculate greedy actions.
            next_actions = torch.argmax(next_q, dim=1, keepdim=True)
            assert next_actions.shape == (self.batch_size, 1)

            # Calculate quantile values of next states and actions at tau_hats.
            next_sa_quantiles = evaluate_quantile_at_action(
                self.target_net.ensemble[i_ens](states=next_states),
                next_actions).transpose(1, 2)
            assert next_sa_quantiles.shape == (self.batch_size, 1, self.N)

            # Calculate target quantile values.
            target_sa_quantiles = rewards[..., None] + (
                1.0 - dones[..., None]) * self.gamma_n * next_sa_quantiles
            assert target_sa_quantiles.shape == (self.batch_size, 1, self.N)

        td_errors = target_sa_quantiles - current_sa_quantiles
        assert td_errors.shape == (self.batch_size, self.N, self.N)

        quantile_huber_loss = calculate_quantile_huber_loss(
            td_errors, self.tau_hats, weights, self.kappa)

        return quantile_huber_loss, next_q.detach().mean().item(), \
            td_errors.detach().abs().sum(dim=1).mean(dim=1, keepdim=True)
