import argparse
import numpy as np


def load(fn, index):
    with open(fn, 'r') as f:
        lines = f.readlines()
    values = {}
    state = False
    for line in lines:
        if line.startswith('Evaluation at training step'):
            state = True
        elif line.startswith('Running experiment'):
            state = False
        elif state:
            terms = line.strip().split(' ')
            name = terms[0]
            acc = float(terms[index])
            if name not in values:
                values[name] = []
            values[name].append(acc)
    return values


def one_experiment(fn, index):
    values = load(fn, index)
    order = ['test', 'visual', 'visual_easier', 'situational_1',
             'situational_2', 'contextual', 'adverb_1', 'adverb_2']
    table = 100 * np.array([values[x] for x in order])
    mean = np.mean(table, -1)
    std = np.std(table, -1)
    return mean, std


def main(fn_list, name_list, args):
    if args.name == 'acc':
        index = -3
    elif args.name == 'target':
        index = -5
    else:
        assert False

    mean_list = []
    std_list = []
    for fn in fn_list:
        mean, std = one_experiment(fn, index)
        mean_list.append(["%.2f" % x for x in mean])
        std_list.append(["%.2f" % x for x in std])

    cols = [name_list]
    for means, stds in zip(mean_list, std_list):
        max_std_length = max([len(x) for x in stds])
        col = []
        for mean, std in zip(means, stds):
            term = mean
            term += ' $\\pm$ '
            for _ in range(max_std_length - len(std)):
                term += '\\:\\:'
            term += std
            col.append(term)
        cols.append(col)

    rows = zip(*cols)
    for row in rows:
        line = ' & '.join(row) + ' \\\\'
        print(line)


if __name__ == '__main__':
    fn_list = [
        'logs/proposed/log.txt',
        'logs/ablation_entreg/log.txt',
        'logs/ablation_input/log.txt',
        'logs/ablation_output/log.txt',
    ]
    name_list = [
        'A: Random',
        'B: Yellow squares',
        'C: Red squares',
        'D: Novel direction',
        'E: Relativity',
        'F: Class inference',
        'G: Adverb $k=1$',
        'H: Adverb to verb',
    ]
    parser = argparse.ArgumentParser()
    parser.add_argument('--name', type=str, default='acc',
                        help='evaluation name')
    args = parser.parse_args()
    main(fn_list, name_list, args)
