"""Data structures for subgroup classification results.

These lightweight containers are *not* intended to provide heavy-duty data
processing but merely to hold the information required by downstream metric
and visualization utilities.
"""
from __future__ import annotations

from abc import ABC, abstractmethod
from typing import Sequence, Union

import numpy as np
from sklearn.isotonic import IsotonicRegression

ArrayLike = Union[Sequence[float], np.ndarray]

class Model(ABC):
    """Abstract base class for binormal data.

    This class defines the minimum interface that binormal implementations
    must provide: access to true labels, predictions, and training prevalence.
    """

    @property
    @abstractmethod
    def train_0(self) -> np.ndarray:
        """Return predictions for class 0 samples."""

    @property
    @abstractmethod
    def train_1(self) -> np.ndarray:
        """Return predictions for class 1 samples."""

    @property
    @abstractmethod
    def train_prevalence(self) -> float:
        """Return the prevalence used during training."""

    def get_fpr_fnr(self, thresholds: ArrayLike | None = None):
        """Compute false-positive and false-negative rates for *thresholds*."""
        if thresholds is None:
            thresholds = np.array([0.5])
        if isinstance(thresholds, list):
            thresholds = np.array(thresholds)
        elif isinstance(thresholds, float):
            thresholds = np.array([thresholds])

        # Classify based on threshold(s).
        fpr = np.mean(self.train_0[:, None] > thresholds[None, :], axis=0)
        fnr = np.mean(self.train_1[:, None] < thresholds[None, :], axis=0)
        return fpr, fnr

    @staticmethod
    def calibrate(scores: np.ndarray, y_true: np.ndarray) -> np.ndarray:
        """Apply isotonic regression to calibrate probability scores.

        Parameters
        ----------
        scores
            Uncalibrated probability predictions
        y_true
            Binary ground-truth labels

        Returns
        -------
        np.ndarray
            Calibrated probability scores
        """
        return IsotonicRegression(out_of_bounds="clip").fit(scores, y_true)
