"""
Centralized seed management system for reproducible results across the project.
This module provides utilities to control random number generation consistently.
"""
from __future__ import annotations

import os
import numpy as np
import random
import hashlib
from typing import Optional, Any, Dict
from contextlib import contextmanager

# Default seed for the project
DEFAULT_SEED = 42

def get_default_seed() -> int:
    """Returns the standard default seed for the project.

    Returns
    -------
    int
        Default seed value.
    """
    env_seed = os.environ.get('BRIER_PREVALENCE_SEED')
    if env_seed is not None:
        try:
            return int(env_seed)
        except ValueError:
            # If env value can't be converted to int, fall back to default
            pass
    return DEFAULT_SEED

def set_global_seed(seed: Optional[int] = None) -> None:
    """Sets seed for all random number generators.

    Parameters
    ----------
    seed : int, optional
        Seed value to use. If None, uses the default seed.
    """
    seed = seed if seed is not None else get_default_seed()
    
    # Set seeds for different libraries
    random.seed(seed)
    np.random.seed(seed)
    
    # Return the used seed for logging purposes
    return seed

def get_derived_seed(base_seed: Optional[int] = None, 
                    component_name: str = "",
                    salt: str = "") -> int:
    """Creates deterministic seed variants from a base seed.

    Parameters
    ----------
    base_seed : int, optional
        Base seed to derive from. If None, uses the default seed.
    component_name : str, default=""
        Name of the component requesting the seed.
    salt : str, default=""
        Additional string to modify the derived seed.

    Returns
    -------
    int
        Deterministic derived seed.
    """
    base_seed = base_seed if base_seed is not None else get_default_seed()
    
    # Create a deterministic hash from the inputs
    seed_data = f"{base_seed}:{component_name}:{salt}"
    hash_obj = hashlib.md5(seed_data.encode())
    
    # Convert first 8 bytes of the hash to an integer
    derived_seed = int.from_bytes(hash_obj.digest()[:8], 'little')
    
    return derived_seed

def get_random_generator(seed: Optional[int] = None, 
                        component_name: str = "") -> np.random.Generator:
    """Creates a NumPy random generator with appropriate seed.

    Parameters
    ----------
    seed : int, optional
        Seed to use. If None, uses the default seed.
    component_name : str, default=""
        Component name for derived seeds.

    Returns
    -------
    np.random.Generator
        NumPy random number generator instance.
    """
    if seed is None:
        seed = get_default_seed()
    
    if component_name:
        seed = get_derived_seed(seed, component_name)
        
    return np.random.default_rng(seed)

@contextmanager
def with_seed(seed: Optional[int] = None):
    """Context manager to temporarily change seed for a code block.

    Parameters
    ----------
    seed : int, optional
        Seed to use within the context. If None, uses the default seed.
    """
    # Store the current state
    python_state = random.getstate()
    numpy_state = np.random.get_state()
    
    # Set the temporary seed
    used_seed = set_global_seed(seed)
    
    try:
        yield used_seed
    finally:
        # Restore the previous state
        random.setstate(python_state)
        np.random.set_state(numpy_state)
