"""
Configuration system for the brier-prevalence-vis package.
Provides a centralized way to manage configuration parameters,
especially those related to reproducibility.
"""
from __future__ import annotations

import os
import json
import logging
from typing import Dict, Any, Optional
from pathlib import Path
import datetime

from reproducibility.seed_manager import get_default_seed

# Set up logging
logger = logging.getLogger(__name__)

# Default configuration values
DEFAULT_CONFIG = {
    "seed": get_default_seed(),
    "output_dir": "./output",
    "log_level": "INFO",
}


class Configuration:
    """Configuration class for the brier-prevalence-vis package."""
    
    def __init__(self, config_dict: Optional[Dict[str, Any]] = None):
        """Initialize with configuration dictionary.

        Parameters
        ----------
        config_dict : Dict[str, Any], optional
            Configuration dictionary to use. If None, uses DEFAULT_CONFIG.
        """
        self._config = DEFAULT_CONFIG.copy()
        if config_dict:
            self._config.update(config_dict)
        
        # Initialize analysis log if needed
        self._analysis_log = []

    def __getitem__(self, key: str) -> Any:
        """Get a configuration value.

        Parameters
        ----------
        key : str
            Configuration key.

        Returns
        -------
        Any
            Configuration value.
        """
        return self._config.get(key)
    
    def __setitem__(self, key: str, value: Any) -> None:
        """Set a configuration value.

        Parameters
        ----------
        key : str
            Configuration key.
        value : Any
            Configuration value.
        """
        self._config[key] = value
    
    @property
    def seed(self) -> int:
        """Get the seed value.

        Returns
        -------
        int
            Seed value.
        """
        # Allow override by environment variable
        env_seed = os.environ.get('BRIER_PREVALENCE_SEED')
        if env_seed is not None:
            try:
                return int(env_seed)
            except ValueError:
                logger.warning(f"Invalid seed value in environment variable: {env_seed}")
        return self._config.get('seed', get_default_seed())
    
    @seed.setter
    def seed(self, value: int) -> None:
        """Set the seed value.

        Parameters
        ----------
        value : int
            Seed value.
        """
        self._config['seed'] = value
    
    def log_analysis(self, analysis_name: str, parameters: Dict[str, Any]) -> None:
        """Log an analysis with its parameters, including the seed used.

        Parameters
        ----------
        analysis_name : str
            Name of the analysis.
        parameters : Dict[str, Any]
            Parameters used for the analysis.
        """
        # Make sure seed is included in the log
        if 'seed' not in parameters:
            parameters['seed'] = self.seed
        
        log_entry = {
            'timestamp': datetime.datetime.now().isoformat(),
            'analysis': analysis_name,
            'parameters': parameters
        }
        
        self._analysis_log.append(log_entry)
        logger.info(f"Analysis logged: {analysis_name} with seed {parameters['seed']}")
    
    def save_analysis_log(self, filename: str = "analysis_log.json") -> None:
        """Save the analysis log to a JSON file.

        Parameters
        ----------
        filename : str, optional
            Filename to save the log to. Default is "analysis_log.json".
        """
        output_dir = Path(self._config.get('output_dir', './output'))
        output_dir.mkdir(exist_ok=True, parents=True)
        output_path = output_dir / filename
        
        with open(output_path, 'w') as f:
            json.dump(self._analysis_log, f, indent=2)
        
        logger.info(f"Analysis log saved to {output_path}")


# Singleton configuration instance
_config_instance = Configuration()


def get_config() -> Configuration:
    """Get the singleton configuration instance.

    Returns
    -------
    Configuration
        The singleton configuration instance.
    """
    return _config_instance
