from sage.all import PolynomialRing, QQ, matrix, vector, ideal

from .border_basis import BorderBasisCalculator


class OBorderBasisChecker(BorderBasisCalculator):
    """
    Class for checking if a given set of polynomials forms an O-border basis.

    Based on Proposition 4 (Buchberger Criterion for Border Bases) from: Computing Border Bases, Kehrein & Kreuzer (2005).

    We check if the given set of generators forms an O-border basis by verifying the following conditions:
    - The ideal generated by the given set of generators is equal to the ideal generated by the set F.
    - All S-polynomials for neighboring generators are reducible.
    """
    def __init__(self, ring):
        """
        Initialize the checker with a polynomial ring and a set of generators.

        :param ring: Polynomial ring
        """
        super().__init__(ring)

    def find_neighbors(self):
        """
        Identify neighboring generators. A pair of generators (g_k, g_l) are neighbors if there respective border terms are related by a pair of variables.

        bl = xi * bk

        or

        xj * bl = xi * bk

        :return: List of neighboring pairs (k, l, xi, xj)
        """
        neighbors = []
        num_generators = len(self.generators)

        border_terms = []
        # determine the border term for each generator
        for g in self.generators:
            for monomial in g.monomials():
                if monomial in self.O_border:
                    border_terms.append(monomial)

        for k in range(num_generators):
            for l in range(num_generators):
                if k < l:  # Avoid redundant pairs

                    bk = border_terms[k]
                    bl = border_terms[l]

                    # Check for neighbor relationships
                    for var1 in self.ring.gens():
                        for var2 in self.ring.gens():
                            if var1 * bk == var2 * bl:
                                neighbors.append((k, l, var1, var2))
                            elif var1 * bk == bl:  # Case x_i * b_k = b_l
                                neighbors.append((k, l, var1, None))

        return list(set(neighbors))

    def s_polynomial(self, k, l, xi, xj=None):
        """
        Compute the S-polynomial for a pair of neighboring generators.

        :param k: Index of the first generator
        :param l: Index of the second generator
        :param xi: Indeterminate corresponding to x_i
        :param xj: Indeterminate corresponding to x_j (if applicable)
        :return: The S-polynomial
        """
        gk = self.generators[k]
        gl = self.generators[l]

        if xj is None:  # Case x_i * b_k = b_l
            #print("S-polynomial:",k,l, xi * gk - gl)
            return xi * gk - gl
        else:  # Case x_i * b_k = x_j * b_l
            #print("S-polynomial:",k,l, xi * gk - xj * gl)
            return xi * gk - xj * gl

    def ideal_check(self, F):
        """
        Check if the given set of generators generates the same ideal as the set F.

        :param F: List of polynomials

        :return: True if the given set of generators generates the same ideal as F, False otherwise
        """
        # Check if the ideal is equal to the ring
        return ideal(F) == ideal(self.generators)

    def check_s_pol_reducability(self):
        """
        Computes all S-polynomials for neighboring generators and checks if they are reducible.

        :return: True if the given set of generators forms an O-border basis, False otherwise
        """
        neighbors = self.find_neighbors()

        # Construct all s-polynomials and check if they are reducible
        for k, l, xi, xj in neighbors:
            s_poly = self.s_polynomial(k, l, xi, xj)

            # apply gaussian elimination to s_poly for reducibility
            result, _ = super().gaussian_elimination(self.generators, [s_poly])

            # If the S-polynomial is not reducible, the set of generators does not form an O-border basis
            if result:
                return False
        return True

    def check_oborder_basis(self, G, O, F):
        """
        Check if the given set of generators forms an O-border basis of the ideal generated by F.

        :param G: Potential O-border basis
        :param O: Order ideal
        :param F: Set of polynomials generating the ideal
        :return: True if G forms an O-border basis, False otherwise
        """
        self.generators = G
        self.O_border = super().border(O)

        # Check if the given set of generators generates the same ideal as F
        ideal_check = self.ideal_check(F)

        # S-polynomial reducibility check
        s_pol_reducibility = self.check_s_pol_reducability()

        # print the results
        print(f"Ideal Check: {ideal_check}")
        print(f"S-polynomial Reducibility: {s_pol_reducibility}")

        # Check both conditions
        return ideal_check and s_pol_reducibility



# Example Usage
if __name__ == "__main__":
    # Define the polynomial ring
    R = PolynomialRing(QQ, 'x, y, z', order='deglex')
    x, y, z = R.gens()

    # Define the border prebasis G
    O = [1, y, x, x*y, y**2]

    F = [
        "x^3 - x",
        "y^3 - y",
        "x^2*y - 1/2*y - 1/2*y^2",
        "x*y - x - 1/2*y + x^2 - 1/2*y^2",
        "x*y^2 - x - 1/2*y + x^2 - 1/2*y^2"
    ]

    F = [R(poly_str) for poly_str in F]


    G = [
        "x**2 + x*y - 1/2*y**2 - x - 1/2*y",
        "y**3 - y",
        "x*y**2 - x*y",
        "x**2*y - 1/2*y**2 - 1/2*y"
    ]

    G = [R(poly_str) for poly_str in G]

    #G =   [y**2 + x - 1, x*y + z**2 - z, y*z**2 - x**2 - y*z + x, y**2*z + x*z - z, x*y*z + z**3 - z**2, x**2*y + x*z**2 - x*z, z**4 + x**3 - 2*z**3 - x**2 + z**2, y*z**3 - x**2*z - x**2 + x*z - y*z + x, x*y*z**2 - x**3 + z**3 + x**2 - z**2, x**2*y*z + x*z**3 - x*z**2, x**3*y + x**2*z**2 - x**2*z, x*z**4 + x**4 - 2*x*z**3 - x**3 + x*z**2, x*y*z**3 - x**3*z - x**3 + x**2*z + z**3 + x**2 - z**2, x**2*y*z**2 - x**4 + x*z**3 + x**3 - x*z**2, x**3*y*z + x**2*z**3 - x**2*z**2, x**4*y + x**3*z**2 - x**3*z, x**2*z**4 + x**5 - 2*x**2*z**3 - x**4 + x**2*z**2, x**2*y*z**3 - x**4*z - x**4 + x**3*z + x*z**3 + x**3 - x*z**2, x**3*y*z**2 - x**5 + x**2*z**3 + x**4 - x**2*z**2, x**4*z**2 - x**4*z + x, x**4*y*z + x**3*z**3 - x**3*z**2, x**5*y - x, x**6 - x**5 - z**2 + z, x**3*z**4 - 2*x**3*z**3 + x**3*z**2 + z**2 - z, x**3*y*z**3 - x**5*z - x**5 + x**4*z + x**2*z**3 + x**4 - x**2*z**2, x**4*z**3 - x**4*z + x*z + x, x**5*z**2 - x**5*z + x**2, x**5*y*z - x*z, x**6*z - x**5*z - z**3 + z**2]



    #O = [1, z, y, x, z**2, x**2, x**3, x**4, x**5, x*z**2, x*z, y*z, z**3, x**2*z, x**3*z, x**4*z, x**5*z, x**2*z**2, x*z**3, x**3*z**2, x**2*z**3, x**3*z**3]


    # Create the checker
    checker = OBorderBasisChecker(R)

    # Check if G forms an O-border basis
    result = checker.check_oborder_basis(G, O, F)
    print(f"Is G an O-border basis? {'Yes' if result else 'No'}")