// server.js - CommonJS style
const express = require("express");
const cors = require("cors");
const dotenv = require("dotenv");
const path = require("path");
const { randomUUID } = require("crypto");
const OpenAI = require("openai");

dotenv.config();

const app = express();
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, "public"))); // serves index.html

const openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });

// --- Helpers ---
const MODELS = {
  fast: "gpt-4o-mini", // quick & cheaper
  quality: "gpt-4o",
};

function clampInt(n, lo, hi) {
  const x = parseInt(n, 10);
  return Math.min(hi, Math.max(lo, isNaN(x) ? lo : x));
}

function rndInt(lo, hi) {
  return Math.floor(Math.random() * (hi - lo + 1)) + lo;
}

async function sayAs(name, partner, cue, theme = "") {
  // One witty, clean line as `name`, reacting to `cue`.
  const sys =
    `You are ${name}, a quick, clean stand-up comedian in a duo with ${partner}.\n` +
    `Keep replies 1–2 sentences, smart, and classroom-safe. If a theme is given, weave it in subtly.`;
  const prompt = [
    theme ? `Theme for the set: ${theme}.` : null,
    cue
      ? `${partner} just said: "${cue}". Reply as ${name}.`
      : `Open the set as ${name} with a crisp opener.`,
  ]
    .filter(Boolean)
    .join("\n");

  const resp = await openai.chat.completions.create({
    model: MODELS.fast,
    temperature: 0.9,
    messages: [
      { role: "system", content: sys },
      { role: "user", content: prompt },
    ],
  });
  return resp.choices?.[0]?.message?.content?.trim() || "";
}

// ---- Comedy: AI vs AI ----
app.post("/api/comedy/duo", async (req, res) => {
  try {
    const { theme = "", rounds = 4, starter = "Joe" } = req.body || {};
    const n = clampInt(rounds, 2, 12);
    let current = starter === "Cathy" ? "Cathy" : "Joe";
    let other = current === "Joe" ? "Cathy" : "Joe";

    let cue = ""; // last partner line
    const transcript = [];

    for (let i = 0; i < n; i++) {
      const line = await sayAs(current, other, cue, theme);
      transcript.push({ speaker: current, text: line });
      cue = line;
      const tmp = current;
      current = other;
      other = tmp;
    }

    res.json({ ok: true, transcript });
  } catch (err) {
    console.error(err);
    res.status(500).json({ ok: false, error: "comedy_duo_failed" });
  }
});

// ---- Comedy: single comedian (one-liner) ----
app.post("/api/comedy/one", async (req, res) => {
  try {
    const { prompt = "", persona = "Joe" } = req.body || {};
    const partner = persona === "Cathy" ? "Joe" : "Cathy";
    const line = await sayAs(persona, partner, prompt, "");
    res.json({ ok: true, line, persona });
  } catch (err) {
    console.error(err);
    res.status(500).json({ ok: false, error: "comedy_one_failed" });
  }
});

// ---- Number Guessing: AI vs AI (binary search + optional banter) ----
app.post("/api/number/ai-vs-ai", async (req, res) => {
  try {
    const { low = 1, high = 100, banter = true } = req.body || {};
    let lo = clampInt(low, -1_000_000, 1_000_000);
    let hi = clampInt(high, lo + 1, 1_000_000);

    const secret = rndInt(lo, hi);
    const steps = [];
    let turns = 0;

    while (lo <= hi && turns < 32) {
      const guess = Math.floor((lo + hi) / 2);

      let feedback = "too low";
      if (guess === secret) feedback = "correct";
      else if (guess > secret) feedback = "too high";

      let quip = "";
      if (banter) {
        const resp = await openai.chat.completions.create({
          model: MODELS.fast,
          temperature: 0.8,
          messages: [
            {
              role: "system",
              content: "You are a witty game show narrator. Short playful lines only.",
            },
            {
              role: "user",
              content: `We are playing guess-my-number. Guess=${guess}, feedback=${feedback}. Say 1 short line.`,
            },
          ],
        });
        quip = resp.choices?.[0]?.message?.content?.trim() || "";
      }

      steps.push({ guess, feedback, quip });
      turns++;

      if (feedback === "correct") {
        break; // inside loop — valid
      }
      if (feedback === "too high") hi = guess - 1;
      else lo = guess + 1;
    }

    res.json({ ok: true, secret, steps });
  } catch (err) {
    console.error(err);
    res.status(500).json({ ok: false, error: "ai_vs_ai_failed" });
  }
});

// ---- Number Guessing: You guess (server stores the secret) ----
const sessions = new Map(); // sessionId -> {secret, attempts, lo, hi, history}

app.post("/api/number/human-start", (req, res) => {
  const { low = 1, high = 100 } = req.body || {};
  const lo = clampInt(low, -1_000_000, 1_000_000);
  const hi = clampInt(high, lo + 1, 1_000_000);
  const secret = rndInt(lo, hi);
  const id = randomUUID();
  sessions.set(id, { secret, attempts: 0, lo, hi, history: [] });
  res.json({ ok: true, sessionId: id, range: { low: lo, high: hi } });
});

app.post("/api/number/human-guess", (req, res) => {
  const { sessionId, guess } = req.body || {};
  const s = sessions.get(sessionId);
  if (!s) return res.status(404).json({ ok: false, error: "bad_session" });

  const g = clampInt(guess, s.lo, s.hi);
  s.attempts++;
  let feedback = "too low";
  if (g === s.secret) feedback = "correct";
  else if (g > s.secret) feedback = "too high";
  s.history.push({ guess: g, feedback });

  if (feedback === "correct") sessions.delete(sessionId);
  res.json({ ok: true, feedback, attempts: s.attempts, history: s.history });
});

// ---- Number Guessing: AI guesses (you provide feedback each turn) ----
app.post("/api/number/ai-next-guess", async (req, res) => {
  try {
    const { low = 1, high = 100, history = [] } = req.body || {};
    let lo = clampInt(low, -1_000_000, 1_000_000);
    let hi = clampInt(high, lo + 1, 1_000_000);

    // Apply feedback to narrow bounds
    for (const { guess, feedback } of history) {
      if (feedback === "too high") hi = Math.min(hi, guess - 1);
      else if (feedback === "too low") lo = Math.max(lo, guess + 1);
      else if (feedback === "correct")
        return res.json({
          ok: true,
          done: true,
          nextGuess: guess,
          line: "Got it!",
        });
    }
    if (lo > hi)
      return res.json({
        ok: true,
        done: true,
        nextGuess: null,
        line: "Inconsistent feedback 🤔",
      });

    const nextGuess = Math.floor((lo + hi) / 2);

    // Short line from AI guesser
    const resp = await openai.chat.completions.create({
      model: MODELS.fast,
      temperature: 0.8,
      messages: [
        {
          role: "system",
          content:
            "You are a confident number-guessing AI. Speak in short, upbeat sentences.",
        },
        {
          role: "user",
          content: `Given narrowed range ${lo}-${hi}, propose the next guess (${nextGuess}) with one short sentence.`,
        },
      ],
    });
    const line =
      resp.choices?.[0]?.message?.content?.trim() || `My guess: ${nextGuess}.`;

    res.json({ ok: true, done: false, nextGuess, line, range: { low: lo, high: hi } });
  } catch (err) {
    console.error(err);
    res.status(500).json({ ok: false, error: "ai_next_guess_failed" });
  }
});

// ---- Health check ----
app.get("/api/health", (req, res) => res.json({ ok: true }));

const PORT = process.env.PORT || 8787;
app.listen(PORT, () =>
  console.log(`Agent demo server running at http://localhost:${PORT}`)
);
