import jax
from jax import numpy as jnp

from ProtLig_GPCRclassA.utils import tf_to_jax, tf_to_jraph_graph_reshape

from ProtLig_GPCRclassA.amino_GNN.make_loss_func import make_loss_func, make_aux_loss_func
from ProtLig_GPCRclassA.amino_GNN.concentration.make_compute_metrics import make_compute_metrics_concentration, make_compute_ec50_metrics

def make_eval_params_step():
    """
    NOTE: state is returned because of state.rng update.
    """
    def eval_step(state, batch):
        state = state.replace(rngs = jax.tree_map(lambda x: jax.random.split(x)[0], state.rngs)) # update PRNGKeys
        logits = state.apply_fn(state.params, batch[:-1], deterministic = True, mutable=['intermediates'])
        return state, logits
    return eval_step
    # return jax.jit(eval_step)


def make_valid_conc_masked_params_epoch(loss_option, logger, # Params version ... TODO: Rename!!!
                        aux_loss_option = None, 
                        loader_output_type = 'jax', num_classes = 3, mask_token_id = None, 
                        cls_token_id = None, pad_token_id = None, sep_token_id = None, 
                        unk_token_id = None, eos_token_id = None, bos_token_id = None,
                        conc_sampler = None,
                        min_conc_sample = -5.0, max_conc_sample = 1.0, step_conc_sample = 0.2, conc_parameter_id_map = None):
    """
    Helper function to create valid_epoch function.
    """
    def get_masked_tokens_and_labels(input_ids, attn_mask, seq_mask_rng):
        seq_label = input_ids.copy()
        seq_mask = jax.random.uniform(seq_mask_rng, shape = input_ids.shape) < 0.15
        cls_mask = input_ids != cls_token_id
        pad_mask = attn_mask.astype(bool)
        sep_mask = input_ids != sep_token_id
        unk_mask = input_ids != unk_token_id
        eos_mask = input_ids != eos_token_id
        bos_mask = input_ids != bos_token_id
        input_ids_mask = seq_mask * cls_mask * pad_mask * sep_mask * unk_mask * eos_mask * bos_mask
        return seq_label, input_ids_mask

    # NOTE: delete loss_func:
    loss_func = make_loss_func(is_weighted = False, option = loss_option, num_classes = num_classes)
    if aux_loss_option is not None:
        aux_loss_func = make_aux_loss_func(option = aux_loss_option)
    else:
        aux_loss_func = None
    compute_metrics = make_compute_metrics_concentration(use_jit = False)
    compute_ec50_metrics = make_compute_ec50_metrics(min_conc_sample, max_conc_sample, conc_parameter_id_map = conc_parameter_id_map)
    eval_step = make_eval_params_step()
    # jit:
    eval_step = jax.jit(eval_step)

    # Case loader outputs jnp.DeviceArray:
    if loader_output_type == 'jax':
        def valid_epoch(state, valid_loader):
            raise NotImplementedError('Logic changed. See tf version of epoch...')
    # Case loader outputs tf.Tensor:
    elif loader_output_type == 'tf':
        def valid_epoch(state, valid_loader):
            batch_metrics = []
            for i, batch in valid_loader.enumerate():
                batch = jax.tree_map(lambda x: jax.device_put(tf_to_jax(x), device = jax.devices()[0]), batch)
                S, G, labels = batch
                G =  tf_to_jraph_graph_reshape(G)
                ec50_label = G.globals['_conc']
                ec50_label['label_mask'] = labels['_main_label']
                # Preprocess concentrations:
                _globals = G.globals.copy()
                _globals.update({'_conc' : G.globals['_conc']['value']})
                G = G._replace(globals = _globals)
                # Get masked tokens:
                input_ids_label, input_ids_mask = get_masked_tokens_and_labels(input_ids = S[2], 
                                                                        attn_mask = S[1], 
                                                                        seq_mask_rng = state.rngs['seq_mask'])
                S = S[:-1] # Discard input_ids
                labels.update({'_input_ids_label' : input_ids_label})
                # Combine back to batch:
                batch = (S, G, labels)
                state, logits = eval_step(state, batch)
                curve_params = logits[1]['intermediates']
                ec50_pred = curve_params['ec50'][0]

                ec50_metrics = compute_ec50_metrics(ec50_pred, ec50_label)

                pred_label = (ec50_pred <= max_conc_sample).astype(jnp.float32)
                metrics = compute_metrics(pred_label, labels = batch[-1])

                metrics.update(ec50_metrics)
                batch_metrics.append(metrics)

            return state, batch_metrics
    return valid_epoch