import jax
from jax import numpy as jnp
import numpy
from scipy.optimize import curve_fit

from ProtLig_GPCRclassA.utils import tf_to_jax, tf_to_jraph_graph_reshape

from ProtLig_GPCRclassA.amino_GNN.make_loss_func import make_loss_func, make_aux_loss_func
from ProtLig_GPCRclassA.amino_GNN.concentration.make_compute_metrics import make_compute_metrics_concentration, make_compute_ec50_metrics

# from Receptor_odorant.JAX.BERT_GNN.CLS_GTransformer.profiling_configs import hparams, _datacase, _h5file, dataparams, datadir, logdir, model, get_profiler_logdir

def make_eval_step():
    """
    NOTE: state is returned because of state.rng update.
    """
    def eval_step(state, batch):
        state = state.replace(rngs = jax.tree_map(lambda x: jax.random.split(x)[0], state.rngs)) # update PRNGKeys
        logits = state.apply_fn(state.params, batch[:-1], deterministic = True)
        return state, logits
    return eval_step
    # return jax.jit(eval_step)


def make_valid_ec50_regression_masked_epoch(loss_option, logger, 
                        aux_loss_option = None, 
                        loader_output_type = 'jax', num_classes = 3, mask_token_id = None, 
                        cls_token_id = None, pad_token_id = None, sep_token_id = None, 
                        unk_token_id = None, eos_token_id = None, bos_token_id = None,
                        ec50_decision_threshold = None,
                        min_conc_sample = -5.0, max_conc_sample = 1.0, step_conc_sample = 0.2, conc_parameter_id_map = None,
                        fit_normalized_curve = False):
    """
    Helper function to create valid_epoch function.
    """
    def get_masked_tokens_and_labels(input_ids, attn_mask, seq_mask_rng):
        seq_label = input_ids.copy()
        seq_mask = jax.random.uniform(seq_mask_rng, shape = input_ids.shape) < 0.15
        cls_mask = input_ids != cls_token_id
        pad_mask = attn_mask.astype(bool)
        sep_mask = input_ids != sep_token_id
        unk_mask = input_ids != unk_token_id
        eos_mask = input_ids != eos_token_id
        bos_mask = input_ids != bos_token_id
        input_ids_mask = seq_mask * cls_mask * pad_mask * sep_mask * unk_mask * eos_mask * bos_mask
        return seq_label, input_ids_mask

    loss_func = make_loss_func(is_weighted = False, option = loss_option, num_classes = num_classes)
    if aux_loss_option is not None:
        aux_loss_func = make_aux_loss_func(option = aux_loss_option)
    else:
        aux_loss_func = None
    compute_metrics = make_compute_metrics_concentration(num_thresholds = 2, use_jit = False) # num_thresholds is just dummy and AUROC and AUPRC are not meaningful.
    compute_ec50_metrics = make_compute_ec50_metrics(min_conc_sample, max_conc_sample, conc_parameter_id_map = conc_parameter_id_map)
    compute_ec50_metrics_true_positive = make_compute_ec50_metrics(min_conc_sample, max_conc_sample, conc_parameter_id_map = conc_parameter_id_map, suffix = '_true_positive')
    eval_step = make_eval_step()
    # jit:
    eval_step = jax.jit(eval_step)

    concentrations = jnp.arange(min_conc_sample, max_conc_sample + step_conc_sample, step_conc_sample)

    # Case loader outputs jnp.DeviceArray:
    if loader_output_type == 'jax':
        def valid_epoch(state, valid_loader):
            raise NotImplementedError('Logic chaged. See tf version of epoch...')
    # Case loader outputs tf.Tensor:
    elif loader_output_type == 'tf':
        def valid_epoch(state, valid_loader):
            batch_metrics = []
            for i, batch in valid_loader.enumerate():
                batch = jax.tree_map(lambda x: jax.device_put(tf_to_jax(x), device = jax.devices()[0]), batch)
                S, G, labels = batch
                G =  tf_to_jraph_graph_reshape(G)

                ec50_labels = {'value' : jnp.squeeze(labels['_main_label'])}
                ec50_labels['label_mask'] = (ec50_labels['value'] < ec50_decision_threshold)
                ec50_labels['parameter'] = conc_parameter_id_map['ec50'] * jnp.ones_like(ec50_labels['value'])

                activity_labels = {'_main_label' : (labels['_main_label'] < ec50_decision_threshold)}

                # Get masked tokens:
                input_ids_label, input_ids_mask = get_masked_tokens_and_labels(input_ids = S[2], 
                                                                        attn_mask = S[1], 
                                                                        seq_mask_rng = state.rngs['seq_mask'])
                S = S[:-1] # Discard input_ids
                
                labels.update({'_input_ids_label' : input_ids_label})
                batch = (S, G, labels)
                state, logits = eval_step(state, batch)
                logits['_main_label'] = jnp.squeeze(logits['_main_label'], axis = -1)

                ec50_metrics = compute_ec50_metrics(logits['_main_label'], ec50_labels)

                # NOTE: Change the ec50_metrics['squared_error_mask'] according to the above WARNING

                label_pred = (logits['_main_label'] < ec50_decision_threshold)                
                metrics = compute_metrics(label_pred.astype(jnp.float32), labels = activity_labels)

                ec50_labels_true_positive = ec50_labels.copy()
                ec50_labels_true_positive['label_mask'] = ec50_labels_true_positive['label_mask'] * label_pred # Only for correctly predicted.
                ec50_metrics_true_positive = compute_ec50_metrics_true_positive(logits['_main_label'], ec50_labels_true_positive)

                metrics.update(ec50_metrics)
                metrics.update(ec50_metrics_true_positive)
                batch_metrics.append(metrics)
            return state, batch_metrics
    return valid_epoch