import os
import random
import time
from dataclasses import dataclass, field

import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
from stable_baselines3.common.atari_wrappers import (
    ClipRewardEnv,
    EpisodicLifeEnv,
    FireResetEnv,
    MaxAndSkipEnv,
    NoopResetEnv,
)
from torch.distributions.categorical import Categorical
import stable_baselines3 as sb3


@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 3
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""

    # Algorithm specific arguments
    env_id: str = "pong_bowling"
    """the environment id of the task"""
    env_names: list = field(default_factory=lambda: ['PongNoFrameskip-v4', 'BowlingNoFrameskip-v4'])
    """the environment names of the task"""
    total_timesteps: int = 3000000
    """total timesteps of the first task"""
    second_timesteps: int = 2000000
    """total timesteps of the second task"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(2e5)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 1.0
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 64  # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 2e4
    """timestep to start learning"""
    policy_lr: float = 3e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 3e-4
    """the learning rate of the Q network network optimizer"""
    update_frequency: int = 4
    """the frequency of training updates"""
    target_network_frequency: int = 8000
    """the frequency of updates for the target networks"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    target_entropy_scale: float = 0.89
    """coefficient for scaling the autotune entropy target"""
    replay_interval: int = 10
    """the interval of replay the previous experience"""
    frozen_size: int = 400
    """the number of skill neurons"""


def make_env(env_id, seed):
    def thunk():
        env = gym.make(env_id)
        env = gym.wrappers.RecordEpisodeStatistics(env)
        env = NoopResetEnv(env, noop_max=30)
        env = MaxAndSkipEnv(env, skip=4)
        env = EpisodicLifeEnv(env)
        if "FIRE" in env.unwrapped.get_action_meanings():
            env = FireResetEnv(env)
        env = ClipRewardEnv(env)
        env = gym.wrappers.ResizeObservation(env, (84, 84))
        env = gym.wrappers.GrayScaleObservation(env)
        env = gym.wrappers.FrameStack(env, 4)

        env.action_space.seed(seed)
        return env

    return thunk


def layer_init(layer, bias_const=0.0):
    nn.init.kaiming_normal_(layer.weight)
    torch.nn.init.constant_(layer.bias, bias_const)
    return layer


def register_hooks(network):
    def save_activation(name):
        def forward_hook(module, input, output):
            with torch.no_grad():
                network.activations[name] = F.relu(output)
        return forward_hook

    def save_gradient(name):
        def backward_hook(module, grad_input, grad_output):
            with torch.no_grad():
                eps = 1e-8
                if name in network.activations:
                    grad_output = grad_output[0]
                    activation = network.activations[name]
                    if grad_output.ndim == 4:
                        neuron_importance = torch.mean(torch.abs(activation * grad_output),
                                                       dim=(0, 2, 3))  # 对每个神经元取平均
                    elif grad_output.ndim == 2:
                        neuron_importance = torch.mean(torch.abs(activation * grad_output), dim=0)  # 对每个神经元取平均
                    # normalization
                    norm_importance = neuron_importance / (
                                torch.sum(neuron_importance) / len(neuron_importance) + eps)

                    if module not in network.importance_weights:
                        network.importance_weights[name] = (1 - network.delta) * norm_importance
                    else:
                        network.importance_weights[name] = network.delta * network.importance_weights[name] + (
                                    1 - self.delta) * norm_importance
        return backward_hook

    # register hook
    network.hook_for_conv1 = network.conv1.register_forward_hook(save_activation('conv1'))
    network.hook_back_conv1 = network.conv1.register_full_backward_hook(save_gradient('conv1'))
    network.hook_for_conv2 = network.conv2.register_forward_hook(save_activation('conv2'))
    network.hook_back_conv2 = network.conv2.register_full_backward_hook(save_gradient('conv2'))
    network.hook_for_conv3 = network.conv3.register_forward_hook(save_activation('conv3'))
    network.hook_back_conv3 = network.conv3.register_full_backward_hook(save_gradient('conv3'))
    network.hook_for_fc1 = network.fc1.register_forward_hook(save_activation('fc1'))
    network.hook_back_fc1 = network.fc1.register_full_backward_hook(save_gradient('fc1'))


def remove_hook(network):
    network.hook_for_conv1.remove()
    network.hook_back_conv1.remove()
    network.hook_for_conv2.remove()
    network.hook_back_conv2.remove()
    network.hook_for_conv3.remove()
    network.hook_back_conv3.remove()
    network.hook_for_fc1.remove()
    network.hook_back_fc1.remove()


# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env, delta=0.9):
        super().__init__()
        obs_shape = env.single_observation_space.shape
        self.conv1 = layer_init(nn.Conv2d(obs_shape[0], 32, kernel_size=8, stride=4))
        self.relu1 = nn.ReLU()
        self.conv2 = layer_init(nn.Conv2d(32, 64, kernel_size=4, stride=2))
        self.relu2 = nn.ReLU()
        self.conv3 = layer_init(nn.Conv2d(64, 64, kernel_size=3, stride=1))
        self.flatten = nn.Flatten()

        with torch.inference_mode():
            x = self.relu1(self.conv1(torch.zeros(1, *obs_shape)))
            x = self.relu2(self.conv2(x))
            output_dim = self.flatten(self.conv3(x)).shape[1]

        self.fc1 = layer_init(nn.Linear(output_dim, 512))
        self.fc_q = layer_init(nn.Linear(512, env.single_action_space.n))

        self.layers = [self.conv1, self.conv2, self.conv3, self.fc1]
        self.activations = {}
        self.importance_weights = {}
        self.delta = delta

    def forward(self, x):
        x1 = self.relu1(self.conv1(x / 255.0))
        x2 = self.relu2(self.conv2(x1))
        x3 = self.conv3(x2)
        x4 = F.relu(self.flatten(x3))
        x5 = F.relu(self.fc1(x4))
        q_vals = self.fc_q(x5)
        return q_vals

    def forward_with_activation(self, x):
        x1 = self.relu1(self.conv1(x / 255.0))
        x2 = self.relu2(self.conv2(x1))
        x3 = self.conv3(x2)
        x4 = F.relu(self.flatten(x3))
        x5 = F.relu(self.fc1(x4))
        q_vals = self.fc_q(x5)
        return q_vals, [x1, x2, x3, x5]

    def frozen(self, top):
        with torch.no_grad():
            for k in range(len(top)):
                if self.layers[k].weight.ndim == 4:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :, :, :] = 0
                        self.layers[k].bias.grad[index] = 0
                else:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :] = 0
                        self.layers[k].bias.grad[index] = 0


class Actor(nn.Module):
    def __init__(self, env, delta=0.9):
        super().__init__()
        obs_shape = env.single_observation_space.shape
        self.conv1 = layer_init(nn.Conv2d(obs_shape[0], 32, kernel_size=8, stride=4))
        self.relu1 = nn.ReLU()
        self.conv2 = layer_init(nn.Conv2d(32, 64, kernel_size=4, stride=2))
        self.relu2 = nn.ReLU()
        self.conv3 = layer_init(nn.Conv2d(64, 64, kernel_size=3, stride=1))
        self.flatten = nn.Flatten()

        with torch.inference_mode():
            x = self.relu1(self.conv1(torch.zeros(1, *obs_shape)))
            x = self.relu2(self.conv2(x))
            output_dim = self.flatten(self.conv3(x)).shape[1]

        self.fc1 = layer_init(nn.Linear(output_dim, 512))
        self.fc_logits = layer_init(nn.Linear(512, env.single_action_space.n))

        self.layers = [self.conv1, self.conv2, self.conv3, self.fc1]
        self.activations = {}
        self.gradients = {}
        self.importance_weights = {}
        self.delta = delta

    def forward(self, x):
        x1 = self.relu1(self.conv1(x))
        x2 = self.relu2(self.conv2(x1))
        x3 = self.conv3(x2)
        x4 = F.relu(self.flatten(x3))
        x5 = F.relu(self.fc1(x4))
        logits = self.fc_logits(x5)
        return logits

    def get_action(self, x):
        logits = self(x / 255.0)
        policy_dist = Categorical(logits=logits)
        action = policy_dist.sample()
        # Action probabilities for calculating the adapted soft-Q loss
        action_probs = policy_dist.probs
        log_prob = F.log_softmax(logits, dim=1)
        return action, log_prob, action_probs
        
    def frozen(self, top):
        with torch.no_grad():
            for k in range(len(top)):
                if self.layers[k].weight.ndim == 4:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :, :, :] = 0
                        self.layers[k].bias.grad[index] = 0
                else:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :] = 0
                        self.layers[k].bias.grad[index] = 0


def train_env_1(envs, device, args, env_name, checkpoint):
    if checkpoint is not None:
        actor_state_dict = torch.load(checkpoint + "/actor_state_dict.pth")
        actor.load_state_dict(actor_state_dict)

        qf1_state_dict = torch.load(checkpoint + "/qf1_state_dict.pth")
        qf1.load_state_dict(qf1_state_dict)

        qf2_state_dict = torch.load(checkpoint + "/qf2_state_dict.pth")
        qf2.load_state_dict(qf2_state_dict)

        qf1_target.load_state_dict(qf1.state_dict())
        qf2_target.load_state_dict(qf2.state_dict())

    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -args.target_entropy_scale * torch.log(1 / torch.tensor(envs.single_action_space.n))
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr, eps=1e-4)
    else:
        alpha = args.alpha

    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the game
    obs, _ = envs.reset(seed=args.seed)
    for global_step in range(args.total_timesteps):
        # ALGO LOGIC: put action logic here
        if checkpoint is None and global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                print(f"{env_name} global_step={global_step}, episodic_return={info['episode']['r']}")
                writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                break

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            if global_step % args.update_frequency == 0:
                data = rb.sample(args.batch_size)
                with torch.no_grad():
                    _, next_state_log_pi, next_state_action_probs = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations)
                    qf2_next_target = qf2_target(data.next_observations)
                    min_qf_next_target = next_state_action_probs * (
                            torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    )
                    # adapt Q-target for discrete Q-function
                    min_qf_next_target = min_qf_next_target.sum(dim=1)
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target)

                qf1_values = qf1(data.observations)
                qf2_values = qf2(data.observations)
                qf1_a_values = qf1_values.gather(1, data.actions.long()).view(-1)
                qf2_a_values = qf2_values.gather(1, data.actions.long()).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss

                # optimize the model
                q_optimizer.zero_grad()
                qf_loss.backward()
                q_optimizer.step()

                _, log_pi, action_probs = actor.get_action(data.observations)
                with torch.no_grad():
                    qf1_values = qf1(data.observations)
                    qf2_values = qf2(data.observations)
                    min_qf_values = torch.min(qf1_values, qf2_values)
                # no need for reparameterization, the expectation can be calculated for discrete actions
                actor_loss = (action_probs * ((alpha * log_pi) - min_qf_values)).mean()

                actor_optimizer.zero_grad()
                actor_loss.backward()
                actor_optimizer.step()

                if args.autotune:
                    alpha_loss = (
                            action_probs.detach() * (-log_alpha.exp() * (log_pi + target_entropy).detach())).mean()

                    a_optimizer.zero_grad()
                    alpha_loss.backward()
                    a_optimizer.step()
                    alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)

    actor_top = eval_neuron(actor, args.frozen_size)
    qf1_top = eval_neuron(qf1, args.frozen_size)
    qf2_top = eval_neuron(qf2, args.frozen_size)

    envs.close()
    return qf1_top, qf2_top, actor_top, rb


def train_env_2(envs, device, args, env_name, qf1_top, qf2_top, actor_top, rb_pre):
    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -args.target_entropy_scale * torch.log(1 / torch.tensor(envs.single_action_space.n))
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr, eps=1e-4)
    else:
        alpha = args.alpha

    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the game
    obs, _ = envs.reset(seed=args.seed)

    for global_step in range(args.total_timesteps):
        # ALGO LOGIC: put action logic here
        if global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                print(f"{env_name} global_step={global_step}, episodic_return={info['episode']['r']}")
                writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                break

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            if global_step % args.update_frequency == 0:
                if global_step % args.replay_interval == 0:
                    data = rb_pre.sample(args.batch_size)
                else:
                    data = rb.sample(args.batch_size)
                with torch.no_grad():
                    _, next_state_log_pi, next_state_action_probs = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations)
                    qf2_next_target = qf2_target(data.next_observations)
                    min_qf_next_target = next_state_action_probs * (
                            torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    )
                    # adapt Q-target for discrete Q-function
                    min_qf_next_target = min_qf_next_target.sum(dim=1)
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target)

                qf1_values = qf1(data.observations)
                qf2_values = qf2(data.observations)
                qf1_a_values = qf1_values.gather(1, data.actions.long()).view(-1)
                qf2_a_values = qf2_values.gather(1, data.actions.long()).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss

                # optimize the model
                q_optimizer.zero_grad()
                qf_loss.backward()
                qf1.frozen(qf1_top)
                qf2.frozen(qf2_top)
                q_optimizer.step()

                _, log_pi, action_probs = actor.get_action(data.observations)
                with torch.no_grad():
                    qf1_values = qf1(data.observations)
                    qf2_values = qf2(data.observations)
                    min_qf_values = torch.min(qf1_values, qf2_values)
                # no need for reparameterization, the expectation can be calculated for discrete actions
                actor_loss = (action_probs * ((alpha * log_pi) - min_qf_values)).mean()

                actor_optimizer.zero_grad()
                actor_loss.backward()
                actor.frozen(actor_top)
                actor_optimizer.step()

                if args.autotune:
                    alpha_loss = (action_probs.detach() * (
                                -log_alpha.exp() * (log_pi + target_entropy).detach())).mean()
                    a_optimizer.zero_grad()
                    alpha_loss.backward()
                    a_optimizer.step()
                    alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)
    envs.close()


def test_env(envs, device, args, env_name):
    obs, _ = envs.reset(seed=args.seed)
    for global_step in range(args.test_timesteps):

        actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
        actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                print(f"test_{env_name} global_step={global_step}, episodic_return={info['episode']['r']}")
                writer.add_scalar(f"test_{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"test_{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                break
        obs = next_obs
    envs.close()


def eval_neuron(model, k):
    layers = ['conv1', 'conv2', 'conv3', 'fc1']
    importance_weights_sum = torch.cat([model.importance_weights[layer] for layer in layers], dim=0)
    top_sum = torch.topk(importance_weights_sum, k, dim=0, largest=True)[1]
    top = [[] for _ in layers]  # Create a list with sublists for each layer
    split_points = [0] + list(
        torch.cumsum(torch.tensor([model.importance_weights[layer].shape[0] for layer in layers]), 0))

    # Distribute the top indices into the appropriate group
    for index in top_sum:
        index = index.item()  # Convert tensor to a Python integer
        for i in range(1, len(split_points)):
            if split_points[i - 1] <= index < split_points[i]:
                top[i - 1].append(index - split_points[i - 1])
                break
    return top


if __name__ == "__main__":
    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.seed)])
            for env_name in args.env_names]

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    q_optimizer = optim.Adam(list(qf1.parameters()) + list(qf2.parameters()), lr=args.q_lr, eps=1e-4)
    actor_optimizer = optim.Adam(list(actor.parameters()), lr=args.policy_lr, eps=1e-4)

    register_hooks(actor)
    register_hooks(qf1)
    register_hooks(qf2)

    qf1_top, qf2_top, actor_top, rb = train_env_1(envs[0], device, args, args.env_names[0], None)
    args.total_timesteps = args.second_timesteps

    remove_hook(actor)
    remove_hook(qf1)
    remove_hook(qf2)

    train_env_2(envs[1], device, args, args.env_names[1], qf1_top, qf2_top, actor_top, rb)

    test_env(envs[0], device, args, args.env_names[0])
    test_env(envs[1], device, args, args.env_names[1])
    writer.close()
