import os
import random
import time
from dataclasses import dataclass, field

import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
from stable_baselines3.common.atari_wrappers import (
    ClipRewardEnv,
    EpisodicLifeEnv,
    FireResetEnv,
    MaxAndSkipEnv,
    NoopResetEnv,
)
from torch.distributions.categorical import Categorical
import stable_baselines3 as sb3

@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 2
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""

    # Algorithm specific arguments
    env_id: str = "pong_bowling"
    """the environment id of the task"""
    env_names: list = field(default_factory=lambda: ['PongNoFrameskip-v4', 'BowlingNoFrameskip-v4'])
    """the environment names of the task"""
    total_timesteps: int = 3000000
    """total timesteps of the first task"""
    second_timesteps: int = 2000000
    """total timesteps of the second task"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(2e5)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 1.0
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 64 # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 2e4
    """timestep to start learning"""
    policy_lr: float = 3e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 3e-4
    """the learning rate of the Q network network optimizer"""
    update_frequency: int = 4
    """the frequency of training updates"""
    target_network_frequency: int = 8000
    """the frequency of updates for the target networks"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    target_entropy_scale: float = 0.89
    """coefficient for scaling the autotune entropy target"""
    replay_interval: int = 10
    """the interval of replay the previous experience"""
    frozen_size: int = 400
    """the number of RL skill neurons"""


def make_env(env_id, seed):
    def thunk():
        env = gym.make(env_id)
        env = gym.wrappers.RecordEpisodeStatistics(env)
        env = NoopResetEnv(env, noop_max=30)
        env = MaxAndSkipEnv(env, skip=4)
        env = EpisodicLifeEnv(env)
        if "FIRE" in env.unwrapped.get_action_meanings():
            env = FireResetEnv(env)
        env = ClipRewardEnv(env)
        env = gym.wrappers.ResizeObservation(env, (84, 84))
        env = gym.wrappers.GrayScaleObservation(env)
        env = gym.wrappers.FrameStack(env, 4)

        env.action_space.seed(seed)
        return env
    return thunk


def layer_init(layer, bias_const=0.0):
    nn.init.kaiming_normal_(layer.weight)
    torch.nn.init.constant_(layer.bias, bias_const)
    return layer

# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env):
        super().__init__()
        obs_shape = env.single_observation_space.shape
        self.conv1 = layer_init(nn.Conv2d(obs_shape[0], 32, kernel_size=8, stride=4))
        self.relu1 = nn.ReLU()
        self.conv2 = layer_init(nn.Conv2d(32, 64, kernel_size=4, stride=2))
        self.relu2 = nn.ReLU()
        self.conv3 = layer_init(nn.Conv2d(64, 64, kernel_size=3, stride=1))
        self.flatten = nn.Flatten()

        with torch.inference_mode():
            x = self.relu1(self.conv1(torch.zeros(1, *obs_shape)))
            x = self.relu2(self.conv2(x))
            output_dim = self.flatten(self.conv3(x)).shape[1]

        self.fc1 = layer_init(nn.Linear(output_dim, 512))
        self.fc_q = layer_init(nn.Linear(512, env.single_action_space.n))
        
        self.layers = [self.conv1, self.conv2, self.conv3, self.fc1]

    def forward(self, x):
        x1 = self.relu1(self.conv1(x / 255.0))
        x2 = self.relu2(self.conv2(x1))
        x3 = self.conv3(x2)
        x4 = F.relu(self.flatten(x3))
        x5 = F.relu(self.fc1(x4))
        q_vals = self.fc_q(x5)
        return q_vals
    
    def forward_with_activation(self, x):
        x1 = self.relu1(self.conv1(x / 255.0))
        x2 = self.relu2(self.conv2(x1))
        x3 = self.conv3(x2)
        x4 = F.relu(self.flatten(x3))
        x5 = F.relu(self.fc1(x4))
        q_vals = self.fc_q(x5)
        return q_vals, [x1, x2, x3, x5]
    
    def frozen(self, top):
        with torch.no_grad():
            for k in range(len(top)):
                if self.layers[k].weight.ndim == 4:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, : , :, :] = 0               
                        self.layers[k].bias.grad[index] = 0
                else:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :] = 0               
                        self.layers[k].bias.grad[index] = 0
                

class Actor(nn.Module):
    def __init__(self, env):
        super().__init__()
        obs_shape = env.single_observation_space.shape
        self.conv1 = layer_init(nn.Conv2d(obs_shape[0], 32, kernel_size=8, stride=4))
        self.relu1 = nn.ReLU()
        self.conv2 = layer_init(nn.Conv2d(32, 64, kernel_size=4, stride=2))
        self.relu2 = nn.ReLU()
        self.conv3 = layer_init(nn.Conv2d(64, 64, kernel_size=3, stride=1))
        self.flatten = nn.Flatten()

        with torch.inference_mode():
            x = self.relu1(self.conv1(torch.zeros(1, *obs_shape)))
            x = self.relu2(self.conv2(x))
            output_dim = self.flatten(self.conv3(x)).shape[1]

        self.fc1 = layer_init(nn.Linear(output_dim, 512))
        self.fc_logits = layer_init(nn.Linear(512, env.single_action_space.n))

        self.layers = [self.conv1, self.conv2, self.conv3, self.fc1]
        
    def forward(self, x):
        x1 = self.relu1(self.conv1(x))
        x2 = self.relu2(self.conv2(x1))
        x3 = self.conv3(x2)
        x4 = F.relu(self.flatten(x3))
        x5 = F.relu(self.fc1(x4))
        logits = self.fc_logits(x5)
        return logits, [x1, x2, x3, x5]

    def get_action(self, x):
        logits, _ = self(x / 255.0)
        policy_dist = Categorical(logits=logits)
        action = policy_dist.sample()
        # Action probabilities for calculating the adapted soft-Q loss
        action_probs = policy_dist.probs
        log_prob = F.log_softmax(logits, dim=1)
        return action, log_prob, action_probs
    
    def get_action_with_activation(self, x):
        logits, activation = self(x / 255.0)
        policy_dist = Categorical(logits=logits)
        action = policy_dist.sample()
        # Action probabilities for calculating the adapted soft-Q loss
        action_probs = policy_dist.probs
        log_prob = F.log_softmax(logits, dim=1)
        return action, log_prob, action_probs, activation

    def frozen(self, top):
        with torch.no_grad():
            for k in range(len(top)):
                if self.layers[k].weight.ndim == 4:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :, :, :] = 0               
                        self.layers[k].bias.grad[index] = 0
                else:
                    for index in top[k]:
                        self.layers[k].weight.grad[index, :] = 0               
                        self.layers[k].bias.grad[index] = 0


def rank_top(masks, eval_episode, k):
    masks_sum = torch.cat(masks, dim=0)
    masks_sum /= eval_episode

    neg_masks = torch.ones_like(masks_sum) - masks_sum
    scores = torch.maximum(neg_masks, masks_sum)
    top_sum = torch.topk(scores, k, dim=0, largest=True)[1]

    # Prepare for dynamically handling mask splits based on shape
    top = [[] for _ in range(len(masks))]
    split_points = [0] + list(torch.cumsum(torch.tensor([mask.shape[0] for mask in masks]), 0))

    # Distribute the top indices into the appropriate group
    for index in top_sum:
        index = index.item()  # Convert tensor to a Python integer
        for i in range(1, len(split_points)):
            if split_points[i - 1] <= index < split_points[i]:
                top[i - 1].append(index - split_points[i - 1])
                break
    return top

def eval_neuron_critic(qf1_features, qf1_values, qf2_features, qf2_values, rb, num, eval_episode, k):
    qf1_masks = []
    qf2_masks = []
    qf1_feature_ave = []
    qf2_feature_ave = []

    for feature in qf1_features:
        qf1_feature_ave.append(feature / num)
        qf1_masks.append(torch.zeros_like(feature))
    for feature in qf2_features:
        qf2_feature_ave.append(feature / num)
        qf2_masks.append(torch.zeros_like(feature))

    qf1_values_ave = qf1_values * 1.0 / num
    qf2_values_ave = qf2_values * 1.0 / num

    for i in range(eval_episode):
        data = rb.sample(1)
        qf1_values, qf1_feature = qf1.forward_with_activation(data.observations)
        qf2_values, qf2_feature = qf2.forward_with_activation(data.observations)
        qf1_a_values = qf1_values.gather(1, data.actions.long()).view(-1)
        qf2_a_values = qf2_values.gather(1, data.actions.long()).view(-1)

        for i in range(len(qf1_feature)):
            if qf1_feature[i].ndim == 4:
                mask = ((qf1_feature[i].cpu().detach().abs().mean(dim=(0, 2, 3)) > qf1_feature_ave[i]).int() == (
                            qf1_a_values[0].cpu().detach() > qf1_values_ave)).float()
                qf1_masks[i] += mask

                mask = ((qf2_feature[i].cpu().detach().abs().mean(dim=(0, 2, 3)) > qf2_feature_ave[i]).int() == (
                            qf2_a_values[0].cpu().detach() > qf2_values_ave)).float()
                qf2_masks[i] += mask

            else:
                mask = ((qf1_feature[i].cpu().detach().abs().mean(dim=(0)) > qf1_feature_ave[i]).int() == (
                            qf1_a_values[0].cpu().detach() > qf1_values_ave)).float()
                qf1_masks[i] += mask

                mask = ((qf2_feature[i].cpu().detach().abs().mean(dim=(0)) > qf2_feature_ave[i]).int() == (
                            qf2_a_values[0].cpu().detach() > qf2_values_ave)).float()
                qf2_masks[i] += mask

    qf1_top = rank_top(qf1_masks, eval_episode, k)
    qf2_top = rank_top(qf2_masks, eval_episode, k)

    return qf1_top, qf2_top


def eval_neuron_actor(base_features, return_sum, envs, num, eval_episode, device, k):
    feature_ave = []
    features = []
    masks = []
    for feature in base_features:
        feature_ave.append(feature / num)
        features.append(torch.zeros_like(feature))
        masks.append(torch.zeros_like(feature))
    return_ave = return_sum / num

    for i in range(eval_episode):
        obs, _ = envs.reset(seed=args.seed)
        done = False
        while not done:
            with torch.no_grad():
                actions, _, _, feature = actor.get_action_with_activation(torch.Tensor(obs).to(device))
                actions = actions.detach().cpu().numpy()

            for i in range(len(feature)):
                if feature[i].ndim == 4:
                    features[i] += feature[i].cpu().detach().abs().mean(dim=(0, 2, 3))
                else:
                    features[i] += feature[i].cpu().detach().abs().mean(dim=(0))

            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            if "final_info" in infos:
                for info in infos["final_info"]:
                    if "episode" not in info:
                        continue
                    done = True
                    for i in range(len(features)):
                        features[i] /= info["episode"]["l"]
                        mask = ((features[i] > feature_ave[i]).int() == torch.Tensor(
                            [info["episode"]["r"] > return_ave])).float()
                        masks[i] += mask[0]

            obs = next_obs

    top = rank_top(masks, eval_episode, k)

    return top


def train_env_1(envs, device, args, env_name, checkpoint):
    if checkpoint is not None:
        actor_state_dict = torch.load(checkpoint + "/actor_state_dict.pth")
        actor.load_state_dict(actor_state_dict)

        qf1_state_dict = torch.load(checkpoint + "/qf1_state_dict.pth")
        qf1.load_state_dict(qf1_state_dict)

        qf2_state_dict = torch.load(checkpoint + "/qf2_state_dict.pth")
        qf2.load_state_dict(qf2_state_dict)

        qf1_target.load_state_dict(qf1.state_dict())
        qf2_target.load_state_dict(qf2.state_dict())

    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -args.target_entropy_scale * torch.log(1 / torch.tensor(envs.single_action_space.n))
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr, eps=1e-4)
    else:
        alpha = args.alpha

    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the game
    obs, _ = envs.reset(seed=args.seed)
    for global_step in range(args.total_timesteps):
        # ALGO LOGIC: put action logic here
        if checkpoint is None and global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                print(f"{env_name} global_step={global_step}, episodic_return={info['episode']['r']}")
                writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                break

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            if global_step % args.update_frequency == 0:
                data = rb.sample(args.batch_size)
                with torch.no_grad():
                    _, next_state_log_pi, next_state_action_probs = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations)
                    qf2_next_target = qf2_target(data.next_observations)
                    min_qf_next_target = next_state_action_probs * (
                            torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    )
                    # adapt Q-target for discrete Q-function
                    min_qf_next_target = min_qf_next_target.sum(dim=1)
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target)

                qf1_values = qf1(data.observations)
                qf2_values = qf2(data.observations)
                qf1_a_values = qf1_values.gather(1, data.actions.long()).view(-1)
                qf2_a_values = qf2_values.gather(1, data.actions.long()).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss

                # optimize the model
                q_optimizer.zero_grad()
                qf_loss.backward()
                q_optimizer.step()

                _, log_pi, action_probs = actor.get_action(data.observations)
                with torch.no_grad():
                    qf1_values = qf1(data.observations)
                    qf2_values = qf2(data.observations)
                    min_qf_values = torch.min(qf1_values, qf2_values)
                # no need for reparameterization, the expectation can be calculated for discrete actions
                actor_loss = (action_probs * ((alpha * log_pi) - min_qf_values)).mean()

                actor_optimizer.zero_grad()
                actor_loss.backward()
                actor_optimizer.step()

                if args.autotune:
                    alpha_loss = (action_probs.detach() * (
                                -log_alpha.exp() * (log_pi + target_entropy).detach())).mean()
                    a_optimizer.zero_grad()
                    alpha_loss.backward()
                    a_optimizer.step()
                    alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)

    if checkpoint is None:
        actor_state_dict = actor.state_dict()
        torch.save(actor_state_dict, f"runs/{run_name}/actor_state_dict.pth")

        qf1_state_dict = qf1.state_dict()
        torch.save(qf1_state_dict, f"runs/{run_name}/qf1_state_dict.pth")

        qf2_state_dict = qf2.state_dict()
        torch.save(qf2_state_dict, f"runs/{run_name}/qf2_state_dict.pth")

    def get_ave_critic():
        obs, _ = envs.reset(seed=args.seed)

        qf1_goals = 0
        qf2_goals = 0
        qf1_features = []
        qf2_features = []
        length = 0

        for step in range(20):
            data = rb.sample(args.batch_size)

            qf1_values, qf1_feature = qf1.forward_with_activation(data.observations)
            qf2_values, qf2_feature = qf2.forward_with_activation(data.observations)
            qf1_a_values = qf1_values.gather(1, data.actions.long()).view(-1)
            qf2_a_values = qf2_values.gather(1, data.actions.long()).view(-1)

            if len(qf1_features) == 0:
                for feature_layer in qf1_feature:
                    if feature_layer.ndim == 4:
                        # Conv layer
                        qf1_features.append(feature_layer.cpu().detach().abs().mean(dim=(0, 2, 3)))
                    else:
                        # Full connected layer
                        qf1_features.append(feature_layer.cpu().detach().abs().mean(dim=(0)))

                for feature_layer in qf2_feature:
                    if feature_layer.ndim == 4:
                        # Conv layer
                        qf2_features.append(feature_layer.cpu().detach().abs().mean(dim=(0, 2, 3)))
                    else:
                        # Full connected layer
                        qf2_features.append(feature_layer.cpu().detach().abs().mean(dim=(0)))
            else:
                for i in range(len(qf1_feature)):
                    if qf1_feature[i].ndim == 4:
                        qf1_features[i] += qf1_feature[i].cpu().detach().abs().mean(dim=(0, 2, 3))
                        qf2_features[i] += qf2_feature[i].cpu().detach().abs().mean(dim=(0, 2, 3))
                    else:
                        qf1_features[i] += qf1_feature[i].cpu().detach().abs().mean(dim=(0))
                        qf2_features[i] += qf2_feature[i].cpu().detach().abs().mean(dim=(0))

            length += args.batch_size
            qf1_goals += torch.sum(qf1_a_values).cpu().detach()
            qf2_goals += torch.sum(qf2_a_values).cpu().detach()

        return qf1_features, qf1_goals, qf2_features, qf2_goals, length

    def get_ave_actor():
        obs, _ = envs.reset(seed=args.seed)
        feature_aves = []
        episode_num = 0
        features = []
        return_sum = 0
        for global_step in range(100000):
            actions, _, _, feature = actor.get_action_with_activation(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

            if len(features) == 0:
                for feature_layer in feature:
                    if feature_layer.ndim == 4:
                        # Conv layer
                        features.append(feature_layer.cpu().detach().abs().mean(dim=(0, 2, 3)))
                    else:
                        # Full connected layer
                        features.append(feature_layer.cpu().detach().abs().mean(dim=(0)))
            else:
                for i in range(len(feature)):
                    if feature[i].ndim == 4:
                        features[i] += feature[i].cpu().detach().abs().mean(dim=(0, 2, 3))
                    else:
                        features[i] += feature[i].cpu().detach().abs().mean(dim=(0))

            # TRY NOT TO MODIFY: execute the game and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # TRY NOT TO MODIFY: record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    if "episode" not in info:
                        continue
                    episode_num += 1
                    return_sum += info["episode"]["r"]
                    if len(feature_aves) == 0:
                        for feature in features:
                            feature_aves.append(feature / info["episode"]["l"])
                    else:
                        for i in range(len(features)):
                            feature_aves[i] += features[i] / info["episode"]["l"]
                    features = []

            # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
            obs = next_obs
        return feature_aves, return_sum, episode_num

    qf1_features, qf1_values, qf2_features, qf2_values, length = get_ave_critic()
    feature_aves, return_sum, episode_num = get_ave_actor()
    qf1_top, qf2_top = eval_neuron_critic(qf1_features, qf1_values, qf2_features, qf2_values, rb, length, 100000,
                                          args.frozen_size)
    actor_top = eval_neuron_actor(feature_aves, return_sum, envs, episode_num, 20, device, args.frozen_size)
    envs.close()
    return qf1_top, qf2_top, actor_top, rb


def train_env_2(envs, device, args, env_name, qf1_top, qf2_top, actor_top, rb_pre):
    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -args.target_entropy_scale * torch.log(1 / torch.tensor(envs.single_action_space.n))
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr, eps=1e-4)
    else:
        alpha = args.alpha

    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the game
    obs, _ = envs.reset(seed=args.seed)
    for global_step in range(args.total_timesteps):
        # ALGO LOGIC: put action logic here
        if global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            with torch.no_grad():
                actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
                actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                print(f"{env_name} global_step={global_step}, episodic_return={info['episode']['r']}")
                writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                break

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            if global_step % args.update_frequency == 0:
                if global_step % args.replay_interval == 0:
                    data = rb_pre.sample(args.batch_size)
                else:
                    data = rb.sample(args.batch_size)
                with torch.no_grad():
                    _, next_state_log_pi, next_state_action_probs = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations)
                    qf2_next_target = qf2_target(data.next_observations)
                    min_qf_next_target = next_state_action_probs * (
                            torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    )
                    # adapt Q-target for discrete Q-function
                    min_qf_next_target = min_qf_next_target.sum(dim=1)
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target)

                qf1_values = qf1(data.observations)
                qf2_values = qf2(data.observations)
                qf1_a_values = qf1_values.gather(1, data.actions.long()).view(-1)
                qf2_a_values = qf2_values.gather(1, data.actions.long()).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss

                # optimize the model
                q_optimizer.zero_grad()
                qf_loss.backward()
                qf1.frozen(qf1_top)
                qf2.frozen(qf2_top)
                q_optimizer.step()

                _, log_pi, action_probs = actor.get_action(data.observations)
                with torch.no_grad():
                    qf1_values = qf1(data.observations)
                    qf2_values = qf2(data.observations)
                    min_qf_values = torch.min(qf1_values, qf2_values)
                # no need for reparameterization, the expectation can be calculated for discrete actions
                actor_loss = (action_probs * ((alpha * log_pi) - min_qf_values)).mean()

                actor_optimizer.zero_grad()
                actor_loss.backward()
                actor.frozen(actor_top)
                actor_optimizer.step()

                if args.autotune:
                    alpha_loss = (action_probs.detach() * (
                                -log_alpha.exp() * (log_pi + target_entropy).detach())).mean()
                    a_optimizer.zero_grad()
                    alpha_loss.backward()
                    a_optimizer.step()
                    alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)
    envs.close()


def test_env(envs, device, args, env_name):
    obs, _ = envs.reset(seed=args.seed)
    for global_step in range(args.test_timesteps):

        actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
        actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if "episode" not in info:
                    continue
                print(f"test_{env_name} global_step={global_step}, episodic_return={info['episode']['r']}")
                writer.add_scalar(f"test_{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"test_{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                break
        obs = next_obs
    envs.close()


if __name__ == "__main__":
    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.seed)]) 
            for env_name in args.env_names]

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    q_optimizer = optim.Adam(list(qf1.parameters()) + list(qf2.parameters()), lr=args.q_lr, eps=1e-4)
    actor_optimizer = optim.Adam(list(actor.parameters()), lr=args.policy_lr, eps=1e-4)

    qf1_top, qf2_top, actor_top, rb  = train_env_1(envs[0], device, args, args.env_names[0], None)
    args.total_timesteps = args.second_timesteps
    train_env_2(envs[1], device, args, args.env_names[1], qf1_top, qf2_top, actor_top, rb)
    
    test_env(envs[0], device, args, args.env_names[0])
    test_env(envs[1], device, args, args.env_names[1])
    writer.close()
