import os
import random
import time
from dataclasses import dataclass, field
import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
import metaworld
from torch.autograd import grad
import stable_baselines3 as sb3


@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 1
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""

    # Algorithm specific arguments
    env_id: str = "window-open-v2_window-close-v2"
    """the environment id of the task"""
    env_names: list = field(default_factory=lambda: ['window-open-v2', 'window-close-v2'])
    """the environment names of the task"""
    total_timesteps: int = 1000000
    """total timesteps of the experiments"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(1e6)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 0.005
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 256  # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 1e4
    """timestep to start learning"""
    policy_lr: float = 5e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 5e-4
    """the learning rate of the Q network network optimizer"""
    policy_frequency: int = 2
    """the frequency of training policy (delayed)"""
    target_network_frequency: int = 2  # Denis Yarats' implementation delays this by 2.
    """the frequency of updates for the target nerworks"""
    noise_clip: float = 0.5
    """noise clip parameter of the Target Policy Smoothing Regularization"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    average_num: int = 100
    """the number of episode to compute average success rate"""
    threshold: float = 0.9
    """the threshold representing finishing training"""


def make_env(env_name, gamma):
    def thunk():
        mt1 = metaworld.MT1(env_name)  # Construct the benchmark, sampling tasks
        env = mt1.train_classes[env_name]()  # Create an environment with task `pick_place`
        task = mt1.train_tasks[0]
        env.set_task(task)  # Set task
        env = gym.wrappers.RecordEpisodeStatistics(env)
        return env

    return thunk


# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env):
        super().__init__()
        self.fc1 = nn.Linear(
            np.array(env.single_observation_space.shape).prod() + np.prod(env.single_action_space.shape), 256)
        self.fc2 = nn.Linear(256, 256)
        self.fc3 = nn.Linear(256, 1)

    def forward(self, x, a):
        x = torch.cat([x, a], 1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = self.fc3(x)
        return x


LOG_STD_MAX = 10
LOG_STD_MIN = -10


class SAC_EWC():
    def __init__(self, actor_, critic1, critic2, rb, device):
        super().__init__()
        self.actor = actor_
        self.critic1 = critic1
        self.critic2 = critic2
        self.replay_buffer = rb
        self.device = device
        self.actor_variables = self.actor.parameters()
        self.critic_variables = list(self.critic1.parameters()) + list(self.critic2.parameters())
        self.all_common_variables = list(self.actor.parameters()) + list(self.critic1.parameters()) + list(
            self.critic2.parameters())
        self.old_params = [param.clone().detach() for param in self.all_common_variables]
        for param in self.old_params:
            param.requires_grad = False
        # Initialize reg_weights with zeros, matching the shape of all_common_variables
        self.reg_weights = [torch.zeros_like(param) for param in self.all_common_variables]

        # Ensure reg_weights are not trainable
        for weight in self.reg_weights:
            weight.requires_grad = False

    def change_task(self):
        for old_param, new_param in zip(self.old_params, self.all_common_variables):
            old_param.data.copy_(new_param.data)
        self._update_reg_weights(self.replay_buffer)

    def _update_reg_weights(self, replay_buffer, batches_num: int = 2, batch_size: int = 256):
        """Calculate importance weights representing how important each weight is for the current task."""
        all_weights = []
        for _ in range(batches_num):
            batch = replay_buffer.sample(batch_size)
            all_weights.append(self._get_importance_weights(batch))

        mean_weights = [torch.mean(torch.stack(weights, dim=0), dim=0) for weights in zip(*all_weights)]

        if self.reg_weights is not None:
            merged_weights = [old_reg + new_reg for old_reg, new_reg in zip(self.reg_weights, mean_weights)]
        else:
            merged_weights = mean_weights

        for old_weight, new_weight in zip(self.reg_weights, merged_weights):
            old_weight.data.copy_(new_weight.data)

    def _get_importance_weights(self, batch):
        actor_mu_gs, actor_std_gs, q1_gs, q2_gs, std = self._get_grads(batch.observations, batch.actions)

        reg_weights = []

        for mu_g, std_g in zip(actor_mu_gs, actor_std_gs):
            if mu_g is None and std_g is None:
                raise ValueError("Both mu and std gradients are None!")
            if mu_g is None:
                mu_g = torch.zeros_like(std_g)
            if std_g is None:
                std_g = torch.zeros_like(mu_g)

            # Broadcasting std for every parameter in the model
            dims_to_add = mu_g.dim() - std.dim()
            # broad_std = std.expand_as(mu_g)
            broad_shape = std.shape + (1,) * dims_to_add
            broad_std = std.view(broad_shape)  # broadcasting

            # Fisher information, see the derivation
            fisher = 1 / (broad_std ** 2 + 1e-6) * (mu_g ** 2 + 2 * std_g ** 2)

            # Sum over the output dimensions
            if fisher.dim() > 1:
                fisher = fisher.sum(dim=1)
            # Clip from below
            fisher = torch.clamp(fisher, min=1e-5)
            # Average over the examples in the batch
            reg_weights.append(fisher.mean(dim=0))

        for q_g in q1_gs:
            fisher = q_g ** 2
            reg_weights.append(fisher.mean(dim=0))

        for q_g in q2_gs:
            fisher = q_g ** 2
            reg_weights.append(fisher.mean(dim=0))

        return reg_weights

    def _get_grads(self, obs, actions):
        # Forward pass
        # obs.requires_grad = True
        mu, std = self.actor.get_mean_std(obs)

        q1 = self.critic1(obs, actions.to(torch.float32))
        q2 = self.critic2(obs, actions.to(torch.float32))

        # Compute gradients with respect to the actor and critic parameters
        actor_mu_gs = grad(mu.sum(), self.actor.parameters(), create_graph=True, allow_unused=True)
        actor_std_gs = grad(std.sum(), self.actor.parameters(), create_graph=True, allow_unused=True)
        q1_gs = grad(q1.sum(), self.critic1.parameters(), create_graph=True, allow_unused=True)
        q2_gs = grad(q2.sum(), self.critic2.parameters(), create_graph=True, allow_unused=True)
        return actor_mu_gs, actor_std_gs, q1_gs, q2_gs, std.sum()

    def get_auxiliary_loss(self):
        """Calculate the regularization loss based on previous parameters and parameter weights."""
        reg_loss = torch.tensor(0.0, device=self.device)
        for new_param, old_param, weight in zip(self.all_common_variables, self.old_params, self.reg_weights):
            diffs = (new_param - old_param) ** 2
            weighted_diffs = weight * diffs
            reg_loss += torch.sum(weighted_diffs)
        return reg_loss


class Actor(nn.Module):
    def __init__(self, env):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod(), 256)
        self.fc2 = nn.Linear(256, 256)
        self.fc_mean = nn.Linear(256, np.prod(env.single_action_space.shape))
        self.fc_logstd = nn.Linear(256, np.prod(env.single_action_space.shape))
        # action rescaling
        self.register_buffer(
            "action_scale", torch.tensor((env.action_space.high - env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.register_buffer(
            "action_bias", torch.tensor((env.action_space.high + env.action_space.low) / 2.0, dtype=torch.float32)
        )

    def forward(self, x):
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        mean = self.fc_mean(x2)
        log_std = self.fc_logstd(x2)
        log_std = torch.tanh(log_std)
        log_std = LOG_STD_MIN + 0.5 * (LOG_STD_MAX - LOG_STD_MIN) * (log_std + 1)  # From SpinUp / Denis Yarats

        return mean, log_std

    def get_action(self, x):
        mean, log_std = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean

    def get_mean_std(self, x):
        mean, log_std = self(x)
        std = log_std.exp()
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return mean, std


def train_env(envs, device, args, env_name, checkpoint, task_index, ewc):
    if checkpoint is None:
        # Automatic entropy tuning
        if args.autotune:
            target_entropy = -torch.prod(torch.Tensor(envs.single_action_space.shape).to(device)).item()
            log_alpha = torch.zeros(1, requires_grad=True, device=device)
            alpha = log_alpha.exp().item()
            a_optimizer = optim.Adam([log_alpha], lr=args.q_lr)
        else:
            alpha = args.alpha
        envs.single_observation_space.dtype = np.float32
        rb = ReplayBuffer(
            args.buffer_size,
            envs.single_observation_space,
            envs.single_action_space,
            device,
            handle_timeout_termination=False,
        )
        start_time = time.time()

        # TRY NOT TO MODIFY: start the game
        obs, _ = envs.reset(seed=args.seed)
        success_total = 0
        episode_total = 0
        end = False
        for global_step in range(args.total_timesteps):
            if end:
                break
            # ALGO LOGIC: put action logic here
            if global_step < args.learning_starts:
                actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
            else:
                actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
                actions = actions.detach().cpu().numpy()

            # TRY NOT TO MODIFY: execute the game and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # TRY NOT TO MODIFY: record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    if info['success'] == 1:
                        success_total += 1
                    episode_total += 1
                    rate = success_total / episode_total
                    if episode_total % args.average_num == 0:
                        if rate > args.threshold:
                            end = True
                        episode_total = success_total = 0
                    print(
                        f"{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                    writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                    writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                    writer.add_scalar(f"{env_name}/charts/success_rate", rate, global_step)

            # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
            real_next_obs = next_obs.copy()
            for idx, trunc in enumerate(truncations):
                if trunc:
                    real_next_obs[idx] = infos["final_observation"][idx]
            rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

            # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
            obs = next_obs

            # ALGO LOGIC: training.
            if global_step > args.learning_starts:
                auxiliary_loss = 0
                if task_index > 1:
                    auxiliary_loss = ewc.get_auxiliary_loss()

                data = rb.sample(args.batch_size)
                with torch.no_grad():
                    next_state_actions, next_state_log_pi, _ = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations, next_state_actions)
                    qf2_next_target = qf2_target(data.next_observations, next_state_actions)
                    min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target).view(-1)

                qf1_a_values = qf1(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
                qf2_a_values = qf2(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss + auxiliary_loss

                # optimize the model
                q_optimizer.zero_grad()
                qf_loss.backward(retain_graph=True)
                q_optimizer.step()

                if global_step % args.policy_frequency == 0:
                    for _ in range(
                            args.policy_frequency
                    ):  # compensate for the delay by doing 'actor_update_interval' instead of 1
                        pi, log_pi, _ = actor.get_action(data.observations)
                        qf1_pi = qf1(data.observations, pi)
                        qf2_pi = qf2(data.observations, pi)
                        min_qf_pi = torch.min(qf1_pi, qf2_pi)
                        actor_loss = ((alpha * log_pi) - min_qf_pi).mean() + auxiliary_loss

                        actor_optimizer.zero_grad()
                        actor_loss.backward(retain_graph=True)
                        actor_optimizer.step()

                        if args.autotune:
                            with torch.no_grad():
                                _, log_pi, _ = actor.get_action(data.observations)
                            alpha_loss = (-log_alpha.exp() * (log_pi + target_entropy)).mean()

                            a_optimizer.zero_grad()
                            alpha_loss.backward()
                            a_optimizer.step()
                            alpha = log_alpha.exp().item()

                # update the target networks
                if global_step % args.target_network_frequency == 0:
                    for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                        target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                    for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                        target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

                if global_step % 100 == 0:
                    writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                    writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                    print("SPS:", int(global_step / (time.time() - start_time)))
                    writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                      global_step)
                    if args.autotune:
                        writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)
    else:
        actor_state_dict = torch.load(checkpoint + "/actor_state_dict.pth")
        actor.load_state_dict(actor_state_dict)

        qf1_state_dict = torch.load(checkpoint + "/qf1_state_dict.pth")
        qf1.load_state_dict(qf1_state_dict)

        qf2_state_dict = torch.load(checkpoint + "/qf2_state_dict.pth")
        qf2.load_state_dict(qf2_state_dict)

        qf1_target.load_state_dict(qf1.state_dict())
        qf2_target.load_state_dict(qf2.state_dict())

        rb = torch.load(checkpoint + "/replay_buffer.pth")
    envs.close()
    return rb


def test_env(envs, device, args, env_name):
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    for global_step in range(args.test_timesteps):
        actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
        actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the game and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    episode_total = success_total = 0
                print(
                    f"test_{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"test_{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"test_{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                writer.add_scalar(f"test_{env_name}/charts/success_rate", rate, global_step)
        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs
    envs.close()


if __name__ == "__main__":
    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.gamma)])
            for env_name in args.env_names]

    max_action = float(envs[0].single_action_space.high[0])

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    q_optimizer = optim.Adam(list(qf1.parameters()) + list(qf2.parameters()), lr=args.q_lr)
    actor_optimizer = optim.Adam(list(actor.parameters()), lr=args.policy_lr)

    rb = train_env(envs[0], device, args, args.env_names[0], None, 1, None)
    ewc = SAC_EWC(actor, qf1, qf2, rb, device)
    ewc.change_task()
    train_env(envs[1], device, args, args.env_names[1], None, 2, ewc)

    test_env(envs[0], device, args, args.env_names[0])
    test_env(envs[1], device, args, args.env_names[1])
    writer.close()
