import os
import random
import time
from dataclasses import dataclass, field
import gymnasium as gym
import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim
import tyro
from stable_baselines3.common.buffers import ReplayBuffer
from torch.utils.tensorboard import SummaryWriter
import metaworld
import stable_baselines3 as sb3


@dataclass
class Args:
    exp_name: str = os.path.basename(__file__)[: -len(".py")]
    """the name of this experiment"""
    seed: int = 5
    """seed of the experiment"""
    torch_deterministic: bool = True
    """if toggled, `torch.backends.cudnn.deterministic=False`"""
    cuda: bool = True
    """if toggled, cuda will be enabled by default"""
    track: bool = False
    """if toggled, this experiment will be tracked with Weights and Biases"""
    wandb_project_name: str = "cleanRL"
    """the wandb's project name"""
    wandb_entity: str = None
    """the entity (team) of wandb's project"""
    capture_video: bool = False
    """whether to capture videos of the agent performances (check out `videos` folder)"""

    # Algorithm specific arguments
    env_id: str = "window-open-v2_window-close-v2"
    """the environment id of the task"""
    env_names: list = field(default_factory=lambda: ['window-open-v2', 'window-close-v2'])
    """the environment id of the task"""
    total_timesteps: int = 1000000
    """total timesteps of the experiments"""
    test_timesteps: int = 100000
    """test timesteps of the experiments"""
    buffer_size: int = int(1e6)
    """the replay memory buffer size"""
    gamma: float = 0.99
    """the discount factor gamma"""
    tau: float = 0.005
    """target smoothing coefficient (default: 0.005)"""
    batch_size: int = 256  # 1280
    """the batch size of sample from the reply memory"""
    learning_starts: int = 1e4
    """timestep to start learning"""
    policy_lr: float = 3e-4
    """the learning rate of the policy network optimizer"""
    q_lr: float = 1e-3
    """the learning rate of the Q network network optimizer"""
    policy_frequency: int = 2
    """the frequency of training policy (delayed)"""
    target_network_frequency: int = 1  # Denis Yarats' implementation delays this by 2.
    """the frequency of updates for the target nerworks"""
    noise_clip: float = 0.5
    """noise clip parameter of the Target Policy Smoothing Regularization"""
    alpha: float = 0.2
    """Entropy regularization coefficient."""
    autotune: bool = True
    """automatic tuning of the entropy coefficient"""
    average_num: int = 100
    """the number of episode to compute average success rate"""
    threshold: float = 0.9
    """the threshold representing finishing training"""
    frozen_size: int = 400
    """the number of skill neurons"""
    replay_interval: int = 10
    """the interval of replay the previous experience"""


def make_env(env_name, gamma):
    def thunk():
        mt1 = metaworld.MT1(env_name)  # Construct the benchmark, sampling tasks
        env = mt1.train_classes[env_name]()  # Create an environment with task `pick_place`
        task = mt1.train_tasks[0]
        env.set_task(task)  # Set task
        env = gym.wrappers.RecordEpisodeStatistics(env)
        return env

    return thunk


def register_hooks(network):
    def save_activation(name):
        def forward_hook(module, input, output):
            network.activations[name] = F.relu(output).to('cpu')
        return forward_hook

    def save_gradient(name):
        def backward_hook(module, grad_input, grad_output):
            eps = 1e-8
            if name in network.activations:
                grad_output = grad_output[0].to('cpu')
                activation = network.activations[name]
                neuron_importance = torch.mean(torch.abs(activation * grad_output), dim=0)  # 对每个神经元取平均
                # normalization
                norm_importance = neuron_importance / (torch.sum(neuron_importance) / len(neuron_importance) + eps)
                if module not in network.importance_weights:
                    network.importance_weights[name] = (1 - network.delta) * norm_importance
                else:
                    network.importance_weights[name] = network.delta * network.importance_weights[
                        name] + (1 - network.delta) * norm_importance
        return backward_hook

    # register hook
    network.fc1.register_forward_hook(save_activation('fc1'))
    network.fc1.register_full_backward_hook(save_gradient('fc1'))
    network.fc2.register_forward_hook(save_activation('fc2'))
    network.fc2.register_full_backward_hook(save_gradient('fc2'))


# ALGO LOGIC: initialize agent here:
class SoftQNetwork(nn.Module):
    def __init__(self, env, delta=0.9):
        super().__init__()
        self.fc1 = nn.Linear(
            np.array(env.single_observation_space.shape).prod() + np.prod(env.single_action_space.shape), 256)
        self.fc2 = nn.Linear(256, 256)
        self.fc3 = nn.Linear(256, 1)
        self.layers = [self.fc1, self.fc2]
        self.activations = {}
        self.importance_weights = {}
        self.delta = delta

    def forward(self, x, a):
        x = torch.cat([x, a], 1)
        x = F.relu(self.fc1(x))
        x = F.relu(self.fc2(x))
        x = self.fc3(x)
        return x

    def frozen(self, top):
        with torch.no_grad():
            # print(self.actor_ln0.weight.shape)
            for k in range(len(top)):
                for index in top[k]:
                    self.layers[k].weight.grad[index, :] = 0
                    self.layers[k].bias.grad[index] = 0


LOG_STD_MAX = 10
LOG_STD_MIN = -10


class Actor(nn.Module):
    def __init__(self, env, delta=0.9):
        super().__init__()
        self.fc1 = nn.Linear(np.array(env.single_observation_space.shape).prod(), 256)
        self.fc2 = nn.Linear(256, 256)
        self.fc_mean = nn.Linear(256, np.prod(env.single_action_space.shape))
        self.fc_logstd = nn.Linear(256, np.prod(env.single_action_space.shape))
        # action rescaling
        self.register_buffer(
            "action_scale", torch.tensor((env.action_space.high - env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.register_buffer(
            "action_bias", torch.tensor((env.action_space.high + env.action_space.low) / 2.0, dtype=torch.float32)
        )
        self.layers = [self.fc1, self.fc2]
        self.activations = {}
        self.gradients = {}
        self.importance_weights = {}
        self.delta = delta

    def forward(self, x):
        x1 = F.relu(self.fc1(x))
        x2 = F.relu(self.fc2(x1))
        mean = self.fc_mean(x2)
        log_std = self.fc_logstd(x2)
        log_std = torch.tanh(log_std)
        log_std = LOG_STD_MIN + 0.5 * (LOG_STD_MAX - LOG_STD_MIN) * (log_std + 1)  # From SpinUp / Denis Yarats

        return mean, log_std, [x1, x2]

    def get_action(self, x):
        mean, log_std, _ = self(x)
        std = log_std.exp()
        normal = torch.distributions.Normal(mean, std)
        x_t = normal.rsample()  # for reparameterization trick (mean + std * N(0,1))
        y_t = torch.tanh(x_t)
        action = y_t * self.action_scale + self.action_bias
        log_prob = normal.log_prob(x_t)
        # Enforcing Action Bound
        log_prob -= torch.log(self.action_scale * (1 - y_t.pow(2)) + 1e-6)
        log_prob = log_prob.sum(1, keepdim=True)
        mean = torch.tanh(mean) * self.action_scale + self.action_bias
        return action, log_prob, mean

    def frozen(self, top):
        with torch.no_grad():
            for k in range(len(top)):
                for index in top[k]:
                    self.layers[k].weight.grad[index, :] = 0
                    self.layers[k].bias.grad[index] = 0


def train_env_1(envs, device, args, env_name, checkpoint):
    if checkpoint is None:
        # Automatic entropy tuning
        if args.autotune:
            target_entropy = -torch.prod(torch.Tensor(envs.single_action_space.shape).to(device)).item()
            log_alpha = torch.zeros(1, requires_grad=True, device=device)
            alpha = log_alpha.exp().item()
            a_optimizer = optim.Adam([log_alpha], lr=args.q_lr)
        else:
            alpha = args.alpha
        envs.single_observation_space.dtype = np.float32
        rb = ReplayBuffer(
            args.buffer_size,
            envs.single_observation_space,
            envs.single_action_space,
            device,
            handle_timeout_termination=False,
        )
        start_time = time.time()

        # TRY NOT TO MODIFY: start the task
        obs, _ = envs.reset(seed=args.seed)
        success_total = 0
        episode_total = 0
        end = False
        for global_step in range(args.total_timesteps):
            if end:
                break
            # ALGO LOGIC: put action logic here
            if global_step < args.learning_starts:
                actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
            else:
                actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
                actions = actions.detach().cpu().numpy()

            # TRY NOT TO MODIFY: execute the task and log data.
            next_obs, rewards, terminations, truncations, infos = envs.step(actions)

            # TRY NOT TO MODIFY: record rewards for plotting purposes
            if "final_info" in infos:
                for info in infos["final_info"]:
                    if info['success'] == 1:
                        success_total += 1
                    episode_total += 1
                    rate = success_total / episode_total
                    if episode_total % args.average_num == 0:
                        if rate > args.threshold:
                            end = True
                        episode_total = success_total = 0
                    print(
                        f"{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                    writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                    writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                    writer.add_scalar(f"{env_name}/charts/success_rate", rate, global_step)

            # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
            real_next_obs = next_obs.copy()
            for idx, trunc in enumerate(truncations):
                if trunc:
                    real_next_obs[idx] = infos["final_observation"][idx]
            rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

            # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
            obs = next_obs

            # ALGO LOGIC: training.
            if global_step > args.learning_starts:
                data = rb.sample(args.batch_size)
                with torch.no_grad():
                    next_state_actions, next_state_log_pi, _ = actor.get_action(data.next_observations)
                    qf1_next_target = qf1_target(data.next_observations, next_state_actions)
                    qf2_next_target = qf2_target(data.next_observations, next_state_actions)
                    min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                    next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                        min_qf_next_target).view(-1)

                qf1_a_values = qf1(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
                qf2_a_values = qf2(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
                qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
                qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
                qf_loss = qf1_loss + qf2_loss

                # optimize the model
                q_optimizer.zero_grad()
                qf_loss.backward()
                q_optimizer.step()

                if global_step % args.policy_frequency == 0:  # TD 3 Delayed update support
                    for _ in range(
                            args.policy_frequency
                    ):  # compensate for the delay by doing 'actor_update_interval' instead of 1
                        pi, log_pi, _ = actor.get_action(data.observations)
                        qf1_pi = qf1(data.observations, pi)
                        qf2_pi = qf2(data.observations, pi)
                        min_qf_pi = torch.min(qf1_pi, qf2_pi)
                        actor_loss = ((alpha * log_pi) - min_qf_pi).mean()

                        actor_optimizer.zero_grad()
                        actor_loss.backward()
                        actor_optimizer.step()

                        if args.autotune:
                            with torch.no_grad():
                                _, log_pi, _ = actor.get_action(data.observations)
                            alpha_loss = (-log_alpha.exp() * (log_pi + target_entropy)).mean()

                            a_optimizer.zero_grad()
                            alpha_loss.backward()
                            a_optimizer.step()
                            alpha = log_alpha.exp().item()

                # update the target networks
                if global_step % args.target_network_frequency == 0:
                    for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                        target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                    for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                        target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

                if global_step % 100 == 0:
                    writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                    writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                    writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                    print("SPS:", int(global_step / (time.time() - start_time)))
                    writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                      global_step)
                    if args.autotune:
                        writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)

        actor_top = eval_neuron(actor, args.frozen_size)
        qf1_top = eval_neuron(qf1, args.frozen_size)
        qf2_top = eval_neuron(qf2, args.frozen_size)

        actor_state_dict = actor.state_dict()
        torch.save(actor_state_dict, f"runs/{run_name}/actor_state_dict.pth")

        qf1_state_dict = qf1.state_dict()
        torch.save(qf1_state_dict, f"runs/{run_name}/qf1_state_dict.pth")

        qf2_state_dict = qf2.state_dict()
        torch.save(qf2_state_dict, f"runs/{run_name}/qf2_state_dict.pth")

        torch.save(rb, f"runs/{run_name}/replay_buffer.pth")
    else:
        actor_state_dict = torch.load(checkpoint + "/actor_state_dict.pth")
        actor.load_state_dict(actor_state_dict)

        qf1_state_dict = torch.load(checkpoint + "/qf1_state_dict.pth")
        qf1.load_state_dict(qf1_state_dict)

        qf2_state_dict = torch.load(checkpoint + "/qf2_state_dict.pth")
        qf2.load_state_dict(qf2_state_dict)

        qf1_target.load_state_dict(qf1.state_dict())
        qf2_target.load_state_dict(qf2.state_dict())

        rb = torch.load(checkpoint + "/replay_buffer.pth")

        actor_top = eval_neuron(actor, args.frozen_size)
        qf1_top = eval_neuron(qf1, args.frozen_size)
        qf2_top = eval_neuron(qf2, args.frozen_size)

    envs.close()
    return qf1_top, qf2_top, actor_top, rb


def train_env_2(envs, device, args, env_name, qf1_top, qf2_top, actor_top, rb_pre):
    # Automatic entropy tuning
    if args.autotune:
        target_entropy = -torch.prod(torch.Tensor(envs.single_action_space.shape).to(device)).item()
        log_alpha = torch.zeros(1, requires_grad=True, device=device)
        alpha = log_alpha.exp().item()
        a_optimizer = optim.Adam([log_alpha], lr=args.q_lr)
    else:
        alpha = args.alpha
    envs.single_observation_space.dtype = np.float32
    rb = ReplayBuffer(
        args.buffer_size,
        envs.single_observation_space,
        envs.single_action_space,
        device,
        handle_timeout_termination=False,
    )
    start_time = time.time()

    # TRY NOT TO MODIFY: start the task
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    end = False
    for global_step in range(args.total_timesteps):
        if end:
            break
        # ALGO LOGIC: put action logic here
        if global_step < args.learning_starts:
            actions = np.array([envs.single_action_space.sample() for _ in range(envs.num_envs)])
        else:
            actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
            actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the task and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    if rate > args.threshold:
                        end = True
                    episode_total = success_total = 0
                print(
                    f"{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                writer.add_scalar(f"{env_name}/charts/success_rate", rate, global_step)

        # TRY NOT TO MODIFY: save data to reply buffer; handle `final_observation`
        real_next_obs = next_obs.copy()
        for idx, trunc in enumerate(truncations):
            if trunc:
                real_next_obs[idx] = infos["final_observation"][idx]
        rb.add(obs, real_next_obs, actions, rewards, terminations, infos)

        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs

        # ALGO LOGIC: training.
        if global_step > args.learning_starts:
            if global_step % args.replay_interval == 0:
                data = rb_pre.sample(args.batch_size)
            else:
                data = rb.sample(args.batch_size)
            with torch.no_grad():
                next_state_actions, next_state_log_pi, _ = actor.get_action(data.next_observations)
                qf1_next_target = qf1_target(data.next_observations, next_state_actions)
                qf2_next_target = qf2_target(data.next_observations, next_state_actions)
                min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - alpha * next_state_log_pi
                next_q_value = data.rewards.flatten() + (1 - data.dones.flatten()) * args.gamma * (
                    min_qf_next_target).view(-1)

            qf1_a_values = qf1(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
            qf2_a_values = qf2(data.observations.to(torch.float32), data.actions.to(torch.float32)).view(-1)
            qf1_loss = F.mse_loss(qf1_a_values, next_q_value)
            qf2_loss = F.mse_loss(qf2_a_values, next_q_value)
            qf_loss = qf1_loss + qf2_loss

            # optimize the model
            q_optimizer.zero_grad()
            qf_loss.backward()
            qf1.frozen(qf1_top)
            qf2.frozen(qf2_top)
            q_optimizer.step()

            if global_step % args.policy_frequency == 0:  # TD 3 Delayed update support
                for _ in range(
                        args.policy_frequency
                ):  # compensate for the delay by doing 'actor_update_interval' instead of 1
                    pi, log_pi, _ = actor.get_action(data.observations)
                    qf1_pi = qf1(data.observations, pi)
                    qf2_pi = qf2(data.observations, pi)
                    min_qf_pi = torch.min(qf1_pi, qf2_pi)
                    actor_loss = ((alpha * log_pi) - min_qf_pi).mean()

                    actor_optimizer.zero_grad()
                    actor_loss.backward()
                    actor.frozen(actor_top)
                    actor_optimizer.step()

                    if args.autotune:
                        with torch.no_grad():
                            _, log_pi, _ = actor.get_action(data.observations)
                        alpha_loss = (-log_alpha.exp() * (log_pi + target_entropy)).mean()

                        a_optimizer.zero_grad()
                        alpha_loss.backward()
                        a_optimizer.step()
                        alpha = log_alpha.exp().item()

            # update the target networks
            if global_step % args.target_network_frequency == 0:
                for param, target_param in zip(qf1.parameters(), qf1_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)
                for param, target_param in zip(qf2.parameters(), qf2_target.parameters()):
                    target_param.data.copy_(args.tau * param.data + (1 - args.tau) * target_param.data)

            if global_step % 100 == 0:
                writer.add_scalar(f"{env_name}/losses/qf1_values", qf1_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_values", qf2_a_values.mean().item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf1_loss", qf1_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf2_loss", qf2_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/qf_loss", qf_loss.item() / 2.0, global_step)
                writer.add_scalar(f"{env_name}/losses/actor_loss", actor_loss.item(), global_step)
                writer.add_scalar(f"{env_name}/losses/alpha", alpha, global_step)
                print("SPS:", int(global_step / (time.time() - start_time)))
                writer.add_scalar(f"{env_name}/charts/SPS", int(global_step / (time.time() - start_time)),
                                  global_step)
                if args.autotune:
                    writer.add_scalar(f"{env_name}/losses/alpha_loss", alpha_loss.item(), global_step)
    envs.close()


def test_env(envs, device, args, env_name):
    obs, _ = envs.reset(seed=args.seed)
    success_total = 0
    episode_total = 0
    for global_step in range(args.test_timesteps):

        actions, _, _ = actor.get_action(torch.Tensor(obs).to(device))
        actions = actions.detach().cpu().numpy()

        # TRY NOT TO MODIFY: execute the task and log data.
        next_obs, rewards, terminations, truncations, infos = envs.step(actions)

        # TRY NOT TO MODIFY: record rewards for plotting purposes
        if "final_info" in infos:
            for info in infos["final_info"]:
                if info['success'] == 1:
                    success_total += 1
                episode_total += 1
                rate = success_total / episode_total
                if episode_total % args.average_num == 0:
                    episode_total = success_total = 0
                print(
                    f"test_{env_name}: global_step={global_step}, episodic_return={info['episode']['r']}, success_rate={rate}")
                writer.add_scalar(f"test_{env_name}/charts/episodic_return", info["episode"]["r"], global_step)
                writer.add_scalar(f"test_{env_name}/charts/episodic_length", info["episode"]["l"], global_step)
                writer.add_scalar(f"test_{env_name}/charts/success_rate", rate, global_step)
        # TRY NOT TO MODIFY: CRUCIAL step easy to overlook
        obs = next_obs
    envs.close()


if __name__ == "__main__":
    if sb3.__version__ < "2.0":
        raise ValueError(
            """Ongoing migration: run the following command to install the new dependencies:
poetry run pip install "stable_baselines3==2.0.0a1"
"""
        )

    args = tyro.cli(Args)
    run_name = f"{args.env_id}__{args.exp_name}__{args.seed}__{int(time.time())}"
    writer = SummaryWriter(f"runs/{run_name}")
    writer.add_text(
        "hyperparameters",
        "|param|value|\n|-|-|\n%s" % ("\n".join([f"|{key}|{value}|" for key, value in vars(args).items()])),
    )

    # TRY NOT TO MODIFY: seeding
    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = args.torch_deterministic

    device = torch.device("cuda" if torch.cuda.is_available() and args.cuda else "cpu")

    # env setup
    envs = [gym.vector.SyncVectorEnv([make_env(env_name, args.gamma)])
            for env_name in args.env_names]

    max_action = float(envs[0].single_action_space.high[0])

    actor = Actor(envs[0]).to(device)
    qf1 = SoftQNetwork(envs[0]).to(device)
    qf2 = SoftQNetwork(envs[0]).to(device)
    qf1_target = SoftQNetwork(envs[0]).to(device)
    qf2_target = SoftQNetwork(envs[0]).to(device)
    qf1_target.load_state_dict(qf1.state_dict())
    qf2_target.load_state_dict(qf2.state_dict())
    q_optimizer = optim.Adam(list(qf1.parameters()) + list(qf2.parameters()), lr=args.q_lr)
    actor_optimizer = optim.Adam(list(actor.parameters()), lr=args.policy_lr)

    register_hooks(actor)
    register_hooks(qf1)
    register_hooks(qf2)

    def eval_neuron(model, k):
        layers = ['fc1', 'fc2']
        importance_weights_sum = torch.cat([model.importance_weights[layer] for layer in layers], dim=0)
        top_sum = torch.topk(importance_weights_sum, k, dim=0, largest=True)[1]
        top = [[] for _ in layers]  # Create a list with sublists for each layer
        split_points = [0] + list(
            torch.cumsum(torch.tensor([model.importance_weights[layer].shape[0] for layer in layers]), 0))

        # Distribute the top indices into the appropriate group
        for index in top_sum:
            index = index.item()  # Convert tensor to a Python integer
            for i in range(1, len(split_points)):
                if split_points[i - 1] <= index < split_points[i]:
                    top[i - 1].append(index - split_points[i - 1])
                    break
        return top

    qf1_top, qf2_top, actor_top, rb = train_env_1(envs[0], device, args, args.env_names[0], None)
    train_env_2(envs[1], device, args, args.env_names[1], qf1_top, qf2_top, actor_top, rb)

    test_env(envs[0], device, args, args.env_names[0])
    test_env(envs[1], device, args, args.env_names[1])
    writer.close()
