
# EquiCAD: A Geometric Equivariant Neural Network for 3D Shape Classification

This project provides a framework for classifying 3D CAD models of industrial parts using Equivariant Neural Networks (ENN) and B-rep (Boundary Representation) data. It supports efficient training, validation, and testing on custom datasets, and includes scripts for multi-class, binary, and letter classification tasks.

## Environment Setup
It is recommended to use a conda environment:
```bash
conda env create -f environment.yml
conda activate EquiCAD
```

## Data Preparation
- The dataset directory should contain `train.txt` and `test.txt`, each listing sample filenames (without extension).
- Each sample must be a DGL `.bin` file. Naming conventions:
    - **cla_part.py**: `category_index.bin` (the part before the underscore is used as the class name)
    - **cla_feature.py**: `1_xxx.bin` or `0_xxx.bin` (prefix indicates binary label)
    - **cla_solid.py**: `A_xxx.bin`, `B_xxx.bin`, ... (first character is the class label for 26 letters)

## Training
```bash
# Industrial part multi-class classification
python cla_part.py train --dataset_path /path/to/your_dataset --batch_size 64 --experiment_name part_cls

# Feature binary classification
python cla_feature.py train --dataset features --dataset_path /path/to/feature_dataset --batch_size 64 --experiment_name feature_cls

# SolidLetters 26-class letter classification
python cla_solid.py train --dataset solidletters --dataset_path /path/to/solidletters --batch_size 64 --experiment_name solid_cls
```
- Logs and model checkpoints are saved in `results/<experiment_name>/<date>/<time>/`.
- Category mapping is automatically saved as `category_mapping.json` (for multi-class tasks).
- Monitor training with Tensorboard:
    ```bash
    tensorboard --logdir results/part_cls
    ```

## Testing & Evaluation
```bash
python cla_part.py test --dataset_path /path/to/your_dataset --checkpoint ./results/part_cls/<date>/<time>/best.ckpt
python cla_feature.py test --dataset features --dataset_path /path/to/feature_dataset --ckpt_dir ./Results/feature_cls/<date>/<time>
python cla_solid.py test --dataset solidletters --dataset_path /path/to/solidletters --checkpoint ./results/solid_cls/<date>/<time>/best.ckpt
```
- Category mapping is loaded automatically for multi-class tasks.
- Test set classification accuracy is reported.
- `cla_feature.py` supports AUC/ROC evaluation and multi-model (ENN/UVNet/PointNet/DGCNN/PointNet2) comparison.

## Script Details
- **cla_part.py**: Industrial part multi-class classification. Automatically scans dataset for unique categories, builds and saves category mapping, and supports dynamic class number.
- **cla_feature.py**: Feature-level binary classification. Supports AUC/ROC evaluation and comparison of multiple models. Binary label is determined by filename prefix.
- **cla_solid.py**: 26-class letter classification for SolidLetters dataset. Label is determined by the first character of the filename.
- All scripts support data centering, normalization, and 90-degree rotation augmentation.

## Main Dependencies
- Python 3.9+
- PyTorch
- DGL
- PyTorch Lightning
- scikit-learn
- tqdm
- scipy
- matplotlib

## Customization & Extension
- Custom datasets are supported as long as `.bin` files follow the naming conventions and `train.txt`/`test.txt` split files are provided.
- See scripts in the `process/` directory for converting STEP and other formats to DGL graphs.

## Acknowledgement & Citation
This project is inspired by the UV-Net equivariant neural network design. If you use this project in your research, please cite the original UV-Net paper:

```
@inproceedings{jayaraman2021uvnet,
 title = {UV-Net: Learning from Boundary Representations},
 author = {Pradeep Kumar Jayaraman and Aditya Sanghi and Joseph G. Lambourne and Karl D.D. Willis and Thomas Davies and Hooman Shayani and Nigel Morris},
 eprint = {2006.10211},
 eprinttype = {arXiv},
 eprintclass = {cs.CV},
 booktitle = {IEEE Conference on Computer Vision and Pattern Recognition (CVPR)},
 year = {2021}
}
```
