import numpy as np


__all__ = [
    'x3_toy',
    'x3_gap_toy',
    'sin_toy',
]


class toy_dataset(object):
    def __init__(self, name=''):
        self.name = name

    def train_samples(self):
        raise NotImplementedError

    def test_samples(self):
        raise NotImplementedError


class x3_toy(toy_dataset):
    def __init__(self, name='x3'):
        self.x_min = -6
        self.x_max = 6
        self.y_min = -100
        self.y_max = 100
        self.confidence_coeff = 3.
        self.f = lambda x: np.power(x, 3)
        self.y_std = 3.
        super(x3_toy, self).__init__(name)

    def train_samples(self, n_data=20):
        np.random.seed(1)

        X_train = np.random.uniform(-4, 4, (n_data, 1))
        epsilon = np.random.normal(0, self.y_std, (n_data, 1))
        y_train = np.squeeze(X_train ** 3 + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-6, 6, num=1000, dtype=np.float32)
        outputs = np.power(inputs, 3)
        return inputs, outputs


class x3_gap_toy(toy_dataset):
    def __init__(self, name='x3_gap'):
        self.x_min = -6
        self.x_max = 6
        self.y_min = -100
        self.y_max = 100
        self.confidence_coeff = 3.
        self.y_std = 3.
        self.f = lambda x: np.power(x, 3)
        super(x3_gap_toy, self).__init__(name)

    def train_samples(self, n_data=20):
        np.random.seed(1)

        X_train_1 = np.random.uniform(-4, -1, (n_data // 2, 1))
        X_train_2 = np.random.uniform(1, 4, (n_data // 2, 1))
        X_train = np.concatenate([X_train_1, X_train_2], axis=0)
        epsilon = np.random.normal(0, self.y_std, (n_data, 1))
        y_train = np.squeeze(X_train ** 3 + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-6, 6, num=1000, dtype=np.float32)
        outputs = np.power(inputs, 3)
        return inputs, outputs


class sin_toy(toy_dataset):
    def __init__(self, name='sin'):
        self.x_min = -5
        self.x_max = 5
        self.y_min = -3.5
        self.y_max = 3.5
        self.confidence_coeff = 1.
        self.y_std = 2e-1

        def f(x):
            return 2 * np.sin(4*x)
        self.f = f
        super(sin_toy, self).__init__(name)

    def train_samples(self):
        np.random.seed(3)

        X_train1 = np.random.uniform(-2, -0.5, (10, 1))
        X_train2 = np.random.uniform(0.5, 2, (10, 1))
        X_train = np.concatenate([X_train1, X_train2], axis=0)
        epsilon = np.random.normal(0, self.y_std, (20, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(self.x_min, self.x_max, 1000)
        return inputs, self.f(inputs)

# class polynomial_toy_2(toy_dataset):
#     def __init__(self, name='polynomial_2'):
#         self.x_min = -10
#         self.x_max = 10
#         self.y_min = -4
#         self.y_max = 3
#         self.confidence_coeff = 3.
#         self.y_std = 3.
#         def f(x):
#             return np.sin(x) + 0.1 * x
#         self.f = f
#         super(polynomial_toy_2, self).__init__(name)
#
#     def train_samples(self):
#         np.random.seed(1)
#
#         X_train1 = np.random.uniform(-9, -7, (20, 1))
#         X_train2 = np.random.uniform(-5, -2, (20, 1))
#         X_train3 = np.random.uniform(2, 5, (20, 1))
#         X_train4 = np.random.uniform(7, 9, (20, 1))
#         X_train = np.concatenate([X_train1, X_train2, X_train3, X_train4], axis=0)
#         epsilon = np.random.normal(0, 0.5, (80, 1))
#         y_train = np.squeeze(self.f(X_train) + epsilon)
#         return X_train, y_train
#
#     def test_samples(self):
#         inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
#         return inputs, self.f(inputs)

class polynomial_toy(toy_dataset):
    def __init__(self, name='polynomial'):
        self.x_min = -10
        self.x_max = 10
        self.y_min = -3
        self.y_max = 13
        self.confidence_coeff = 3.
        self.y_std = 3.

        def f(x):
            return np.sin(x) + 0.1 * x ** 2

        self.f = f
        super(polynomial_toy, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-7.5, -1.5, (50, 1))
        X_train2 = np.random.uniform(1.5, 7.5, (50, 1))
        X_train = np.concatenate([X_train1, X_train2], axis=0)
        epsilon = np.random.normal(0, 0.5, (100, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)

class polynomial_gap_toy(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -10
        self.x_max = 10
        self.y_min = -3
        self.y_max = 13
        self.confidence_coeff = 3.
        self.y_std = 3.

        def f(x):
            return np.sin(x) + 0.1 * x ** 2

        self.f = f
        super(polynomial_gap_toy, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-7.5, -2.5, (10, 1)) #(-5, -7.5)
        X_train2 = np.random.uniform(5, 7.5, (10, 1))
        X_train = np.concatenate([X_train1, X_train2], axis=0)
        epsilon = np.random.normal(0, 0.5, (20, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)

class polynomial_gap_toy2(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -10
        self.x_max = 10
        self.y_min = -3
        self.y_max = 13
        self.confidence_coeff = 3.
        self.y_std = 3.

        def f(x):
            return np.sin(x) + 0.1 * x ** 2

        self.f = f
        super(polynomial_gap_toy2, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-7.5, -5, (30, 1))
        X_train2 = np.random.uniform(5, 7.5, (30, 1))
        X_train3 = np.random.uniform(-1.5, 1, (30, 1))
        X_train = np.concatenate([X_train1, X_train2, X_train3], axis=0)
        epsilon = np.random.normal(0, 0.5, (90, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)

class polynomial_gap_toy3(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -10
        self.x_max = 10
        self.y_min = -3
        self.y_max = 13
        self.confidence_coeff = 3.
        self.y_std = 3.

        def f(x):
            return np.sin(x) + 0.1 * x ** 2

        self.f = f
        super(polynomial_gap_toy3, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-7.5, -5.5, (10, 1))
        X_train2 = np.random.uniform(5, 6.5, (10, 1))
        X_train3 = np.random.uniform(-1.5, 1, (10, 1))
        X_train = np.concatenate([X_train1, X_train2, X_train3], axis=0)
        epsilon = np.random.normal(0, 0.5, (30, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)

class polynomial_toy_2(toy_dataset):
    def __init__(self, name='polynomial'):
        self.x_min = -10
        self.x_max = 10
        self.y_min = -4
        self.y_max = 3
        self.confidence_coeff = 3.
        self.y_std = 3.

        def f(x):
            return np.sin(x) + 0.1 * x

        self.f = f
        super(polynomial_toy_2, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-9, -7, (20, 1))
        X_train2 = np.random.uniform(-5, -2, (20, 1))
        X_train3 = np.random.uniform(2, 5, (20, 1))
        X_train4 = np.random.uniform(7, 9, (20, 1))
        X_train = np.concatenate([X_train1, X_train2, X_train3, X_train4], axis=0)
        epsilon = np.random.normal(0, 0.5, (80, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)

class g2_toy(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -10
        self.x_max = 10
        self.y_min = -3
        self.y_max = 3
        self.confidence_coeff = 3.
        self.y_std = 1.

        def f(x):
            return np.sin(x) + 0.1 * x

        self.f = f
        super(g2_toy, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-7.5, -5, (10, 1)) #(-5, -7.5)
        X_train2 = np.random.uniform(-2.5, 2.5, (10, 1))
        X_train3 = np.random.uniform(5, 7.5, (10, 1))
        X_train = np.concatenate([X_train1,  X_train2, X_train3], axis=0)
        epsilon = np.random.normal(0, 0.5, (30, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-10, 10, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)


class g3_toy(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -1
        self.x_max = 1
        self.y_min = -2
        self.y_max = 2
        self.confidence_coeff = 3.
        self.y_std = 1.  #1.

        def f(x):
            return np.sin(3 * 3.14 * x) + 0.3 * np.cos(9 * 3.14 * x) + 0.5 * np.sin(7 * 3.14 * x)

        self.f = f
        super(g3_toy, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(-0.75, -0.25, (10, 1)) #(-5, -7.5)
        X_train2 = np.random.uniform(0.25, 0.75, (10, 1))
        # X_train3 = np.random.uniform(5, 7.5, (10, 1))
        X_train = np.concatenate([X_train1, X_train2], axis=0)
        epsilon = np.random.normal(0, 0.25, (20, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-1, 1, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)


class prior_toy(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -13
        self.x_max = 13
        self.y_min = -5
        self.y_max = 5
        self.confidence_coeff = 3.
        self.y_std = 1.  # 1.
        super(prior_toy, self).__init__(name)


    def test_samples(self):
        inputs = np.linspace(-13, 13, num=1000, dtype=np.float32)
        return inputs, inputs

class isotropy_toy(toy_dataset):
    def __init__(self, name='polynomial_gap'):
        self.x_min = -0.2
        self.x_max = 1.2
        self.y_min = -0.2
        self.y_max = 1.2
        self.confidence_coeff = 3.
        self.y_std = 1.  #1.

        def f(x):
            return x + 0.3 * np.sin(2 * 3.14 * x) + 0.3 * np.sin(4 * 3.14 * x)

        self.f = f
        super(isotropy_toy, self).__init__(name)

    def train_samples(self):
        np.random.seed(1)

        X_train1 = np.random.uniform(0, 0.3, (10, 1)) #(-5, -7.5)
        X_train2 = np.random.uniform(0.3, 0.7, (10, 1))
        # X_train3 = np.random.uniform(5, 7.5, (10, 1))
        X_train = np.concatenate([X_train1, X_train2], axis=0)
        epsilon = np.random.normal(0, 0.02, (20, 1))
        y_train = np.squeeze(self.f(X_train) + epsilon)
        return X_train, y_train

    def test_samples(self):
        inputs = np.linspace(-0.5, 1.5, num=1000, dtype=np.float32)
        return inputs, self.f(inputs)
