# Copyright 2022 The HuggingFace Datasets Authors and the current dataset script contributor.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Perplexity Metric."""
import os
import datasets
import numpy as np
import torch
from torch.nn import CrossEntropyLoss
from transformers import AutoModelForCausalLM, AutoTokenizer

import evaluate
from evaluate import logging
from accelerate import Accelerator
import time
from .utils import add_bos_token, print_cuda_max_memory

_CITATION = """\

"""

_DESCRIPTION = """
Perplexity (PPL) is one of the most common metrics for evaluating language models.
It is defined as the exponentiated average negative log-likelihood of a sequence, calculated with exponent base `e`.

For more information, see https://huggingface.co/docs/transformers/perplexity
"""

_KWARGS_DESCRIPTION = """
Args:
    model_id (str): model used for calculating Perplexity
            NOTE: Perplexity can only be calculated for causal language models.
                    This includes models such as gpt2, causal variations of bert,
                    causal versions of t5, and more (the full list can be found
                    in the AutoModelForCausalLM documentation here:
                    https://huggingface.co/docs/transformers/master/en/model_doc/auto#transformers.AutoModelForCausalLM )

    predictions (list of str): input text, each separate text snippet
        is one list entry.
    batch_size (int): the batch size to run texts through the model. Defaults to 16.
    add_start_token (bool): whether to add the start token to the texts,
        so the perplexity can include the probability of the first word. Defaults to True.
    device (str): device to run on, defaults to 'cuda' when available
Returns:
    perplexity: dictionary containing the perplexity scores for the texts
        in the input list, as well as the mean perplexity. If one of the input texts is
        longer than the max input length of the model, then it is truncated to the
        max length for the perplexity computation.
"""
class Perplexity(evaluate.Metric):

    def _info(self):
        return evaluate.MetricInfo(
            module_type="metric",
            description=_DESCRIPTION,
            citation=_CITATION,
            inputs_description=_KWARGS_DESCRIPTION,
            features=datasets.Features(
                {
                    "predictions": datasets.Value("string"),
                }
            ),
            reference_urls=[
                "https://huggingface.co/docs/transformers/perplexity"],
        )
    
    def _compute_ppx(self, shift_labels, shift_logits, shift_attention_mask_batch):
        loss_fct = CrossEntropyLoss(reduction="none")
        perplexity_batch = torch.exp(
            (loss_fct(shift_logits.transpose(1, 2), shift_labels) * shift_attention_mask_batch).sum(1)
            / shift_attention_mask_batch.sum(1)
        )
        return perplexity_batch.tolist()

    def _eval_gen(sel, model, encoded_batch, past_key_values, out_logits_gen, generation):
        for i in range(generation):
            # Select the current token
            if -generation+i+1 == 0:
                encoded_batch_con = encoded_batch[:, -generation+i:]
            else:
                encoded_batch_con = encoded_batch[:, -generation+i:-generation+i+1]
            # Generate output with kv cache
            outputs = model(encoded_batch_con, past_key_values=past_key_values) 
            # Update past key-value states
            past_key_values = outputs.past_key_values
            
            out_logits_gen = torch.cat((out_logits_gen[..., -generation:, :], outputs.logits), dim=1)
        print_cuda_max_memory()
        return out_logits_gen, past_key_values

    def _compute(
        self, predictions, model, model_id, tokenizer, 
        batch_size: int = 16, add_start_token: bool = True, device=None, max_length=None,
        generation=-1, torch_dtype = torch.float32, eval_standard=False, debug_mode=False
    ):

        if device is not None:
            assert device in ["gpu", "cpu", "cuda"], "device should be either gpu or cpu."
            if device == "gpu":
                device = "cuda"
        else:
            device = "cuda" if torch.cuda.is_available() else "cpu"

        # if batch_size > 1 (which generally leads to padding being required), and
        # if there is not an already assigned pad_token, assign an existing
        # special token to also be the padding token
        if tokenizer.pad_token is None and batch_size > 1:
            existing_special_tokens = list(tokenizer.special_tokens_map_extended.values())
            # check that the model already has at least one special token defined
            assert (
                len(existing_special_tokens) > 0
            ), "If batch_size > 1, model must have at least one special token to use for padding. Please use a different model or set batch_size=1."
            # assign one of the special tokens to also be the pad token
            tokenizer.add_special_tokens({"pad_token": existing_special_tokens[0]})
        if batch_size > 1:
            padding = True
        else: 
            padding = False

        if add_start_token and max_length:
            # leave room for <BOS> token to be added:
            assert (
                tokenizer.bos_token is not None
            ), "Input model must already have a BOS token if using add_start_token=True. Please use a different model, or set add_start_token=False"
            max_tokenized_len = max_length - 1
        else:
            max_tokenized_len = max_length

        # check that each input is long enough:
        # if add_start_token:
        #     assert torch.all(torch.ge(attn_masks.sum(1), 1)), "Each input text must be at least one token long."
        # else:
        #     assert torch.all(
        #         torch.ge(attn_masks.sum(1), 2)
        #     ), "When add_start_token=False, each input text must be at least two tokens long. Run with add_start_token=True if inputting strings of only one token, and remove all empty input strings."

        full_ppls = []
        gen_ppls = []
        full_ppls_sub = []
        gen_ppls_sub = []

        for start_index in logging.tqdm(range(0, len(predictions), batch_size)):
            end_index = min(start_index + batch_size, len(predictions))
            encodings = tokenizer(
                predictions[start_index:end_index],
                add_special_tokens=False,
                padding=padding,
                truncation=True if max_tokenized_len else False,
                max_length=max_tokenized_len,
                return_tensors="pt",
                return_attention_mask=True,
            ).to(device)

            encoded_batch = encodings["input_ids"]
            attn_mask = encodings["attention_mask"]
            print(encoded_batch.shape)

            if add_start_token:
                encoded_batch, attn_mask = add_bos_token(tokenizer, encoded_batch, attn_mask, device)

            labels = encoded_batch
            with torch.no_grad():
                shift_labels = labels[..., 1:].contiguous()
                shift_attention_mask_batch = attn_mask[..., 1:].contiguous()

                if eval_standard:
                    out_logits = model(encoded_batch, attention_mask=attn_mask).logits # [b, n, dic_size]
                    print("standard: torch.cuda.max_memory_allocated: %fGB"%(torch.cuda.max_memory_allocated(0)/1024/1024/1024))
                    shift_logits = out_logits[..., :-1, :].contiguous().to(torch_dtype)
                    if not debug_mode:
                        full_ppls.append(self._compute_ppx(shift_labels, shift_logits, shift_attention_mask_batch))
                    if generation > 0:
                        full_ppls_sub.append(self._compute_ppx(shift_labels[..., -generation:], shift_logits[..., -generation:, :], shift_attention_mask_batch[..., -generation:]))
                    del out_logits, shift_logits
                
                if generation > 0:
                    encoded_batch_init = encoded_batch[:,:-generation]
                    attn_mask_init = attn_mask[:,:-generation]
                    outputs = model(encoded_batch_init, attention_mask=attn_mask_init)
                    past_key_values = outputs.past_key_values
                    out_logits_gen = outputs.logits
                    if not debug_mode:
                        print(tokenizer.decode(encoded_batch[0, -generation:]))
                        out_logits_gen, past_key_values = self._eval_gen(model, encoded_batch, past_key_values, out_logits_gen, generation)


                    file_name = f'{model_id.split("/")[-1]}-paul_cache_{encoded_batch_init.shape[1]}_{generation}.pt'
                    # file_path = os.path.join("tmp/cache", file_name)
                    # torch.save({'past_key_values':past_key_values}, file_path)
                
                if generation > 0 and not debug_mode:
                    shift_logits_gen = out_logits_gen[..., :-1, :].contiguous().to(torch.float32) 
                    shift_labels = shift_labels
                    shift_attention_mask_batch = shift_attention_mask_batch
                    # gen_ppls.append(self._compute_ppx(shift_labels, shift_logits_gen, shift_attention_mask_batch))
                    gen_ppls_sub.append(self._compute_ppx(shift_labels[..., -generation:], shift_logits_gen[..., -generation:, :], shift_attention_mask_batch[..., -generation:]))
            

        if eval_standard:
            if not debug_mode:
                print('full      mean_perplexity: ', round(np.mean(full_ppls), 2))
            if generation > 0:
                print('full_sub  mean_perplexity: ', round(np.mean(full_ppls_sub), 2))
        if generation > 0 and not debug_mode:
            # print('gen       mean_perplexity: ', round(np.mean(gen_ppls), 2))
            print('gen_sub   mean_perplexity: ', round(np.mean(gen_ppls_sub), 2))
            return round(np.mean(gen_ppls_sub), 2)
