"""
Utility module for generating specialized signal sequences and eigenvalue patterns.

This module provides functions for creating structured signals and eigenvalue sequences
used throughout the experimental framework. These sequences follow specific mathematical
patterns that are critical to the paper's theoretical results.

Key Functions:
- generate_lambda(d, gamma): Generates eigenvalues with polynomial decay: lambda[i] = i^(-gamma).
- generate_theta3(q, p, J, d, sigma): Creates a signal with background noise and specific
  signal values at particular indices determined by the q-misalignment parameter.
"""

import numpy as np



def generate_lambda(d, gamma):
    """
    Generate a d-dimensional vector of eigenvalues with polynomial decay.
    
    Creates a vector β where β[i] = i^(-gamma), representing eigenvalues
    with polynomial decay controlled by the gamma parameter.
    
    Args:
        d (int): Vector dimension.
        gamma (float): Decay rate exponent.
        
    Returns:
        beta (numpy.ndarray): Generated d-dimensional vector of eigenvalues.
    """
    i = np.arange(1, d + 1, dtype = float)
    beta = i**(-gamma)
    return beta



def generate_theta3(q, p, J, d, sigma):
    """
    Generate a signal vector with q-misaligned structure.
    
    Creates a d-dimensional vector with background noise and carefully positioned
    signal components. The q parameter controls the misalignment between signal
    location and eigenvalue magnitudes.
    
    Args:
        q (float): Misalignment parameter - controls index spacing as j^q.
        p (float): Signal strength parameter - controls decay as j^(-(p+1)/2).
        J (int): Maximum number of signal components.
        d (int): Vector dimension.
        sigma (float): Standard deviation of background noise.
        
    Returns:
        theta (numpy.ndarray): Generated d-dimensional signal vector.
    """
    # Initialize d-dimensional vector with background noise
    theta = np.ones(d) * sigma / 5
    
    # Iterate through possible values of j up to J
    j = 1
    while j <= J:
        index = int(j**q) - 1  # Calculate index (Python indices start at 0)
        theta[index] = 5 * j**(-(p+1)/2)  # Set value at theta[j^q]
        j += 1
    
    return theta
