import math
import scipy.io
import numpy as np
import scipy.sparse as sp
from scipy.linalg import eigh
#Compute the Maximal Allowable Variance Proxy - Subsection 4.1 for A = BCSSTK09

#Step 1: Load the matrix A = BCSSTK09
# Load the .mat file
mat = scipy.io.loadmat('bcsstk09.mat')

# Access the matrix data
data = mat['Problem']['A'][0,0] # Access the actual sparse matrix

# Convert the sparse matrix to a dense NumPy array
data = data.toarray().astype(np.float64)

# Now compute its singular values
_, s, _ = np.linalg.svd(data, full_matrices=False)
print(s)
# Compute the Stable rank of Inverse
r = sum(s[-1] / sv for sv in s)

print(f"Stable rank of Inverse = {r}")
# Print the least singular value
least_singular_value = s[-1]  # Singular values are sorted in descending order
print(f"Least Singular Value: {least_singular_value}")

#Step 2: Compute the low-rank parameter p such that the spectral tail < 0.05.
# Evaluate \Delta^{\max} for each 1 \leq p' \leq p as described in Section 4.1

def analyze_matrix(A):
    n = A.shape[0]

    # Step 1: Compute eigenvalues in increasing order: λ_n < ... < λ_1
    eigvals = eigh(A, eigvals_only=True)
    eigvals = np.sort(eigvals)  # eigvals[0] = λ_n, eigvals[n-1] = λ_1

    # Re-index as λ_{n-i} = eigvals[i], where i = 0 (→ λ_n), ..., n-1 (→ λ_1)
    invs = 1.0 / eigvals
    total = np.sum(invs)

    # Task 1: Find smallest p s.t. tail sum / total sum < 0.05
    p = None
    lambda_n = eigvals[0]
    for i in range(1, n):
        if lambda_n / eigvals[i] < 0.05:
            p = i
            break

    if p is None:
       print("Task 1: No such p found where λ_n / λ_{n-p} < 0.05")
       return

    print(f"Task 1: Smallest p such that λ_n / λ_(n-p) < 0.05 is p = {p}")

    # Task 2: δ_{n-i} = λ_{n-i} - λ_{n-i-1} = eigvals[i] - eigvals[i-1]
    print("\nTask 2: Gaps δ_{n-i} for 0 ≤ i ≤ p:")
    for i in range(p + 1):
        if i == 0:
            print(f"δ_{{n-{i}}} = undefined (no λ_{{n-{i+1}}})")
        else:
            delta = eigvals[i] - eigvals[i - 1]
            print(f"δ_{{n-{i}}} = λ_{{{i}}} - λ_{{{i-1}}} = {delta:.4e}")

    # Task 3: min{λ_n, δ_{n-i}} / (8 √n)
    print("\nTask 3: min{λ_n, δ_{n-i}} / (8√n) for 0 ≤ i ≤ p:")
    lambda_n = eigvals[0]
    for i in range(p + 1):
        if i == 0:
            print(f"Ratio_{{n-{i}}} = undefined (no δ_{{n-{i}}})")
        else:
            delta = eigvals[i] - eigvals[i - 1]
            val = min(lambda_n, delta) / (8 * np.sqrt(n))
            print(f"Ratio_{{n-{i}}} = min(λ_n, δ_{{n-{i}}}) / (8√n) = {val:.4e}")

analyze_matrix(data)