from qiskit_aer.noise import NoiseModel
from qiskit_aer.noise import depolarizing_error, thermal_relaxation_error, ReadoutError

__all__ = ["noise_model1", "noise_model2"]


# Default noise model
# The default parameters are obtained from these papers:
# * The longest T1 https://doi.org/10.1038/s41467-021-22030-5
# * The longest T2 https://arxiv.org/pdf/1905.13641.pdf
# * Gate length https://www.nature.com/articles/s41534-016-0004-0
# * Readout error
def noise_model1(
    depolarizing_lambda: float = 1e-4,
    t1: float = 1000,
    t2: float = 100,
    single_q_gate_time: float = 10,
    two_q_gate_time: float = 10,
    three_q_gate_time: float = 1,
    readout_p00: float = 1.0,
    readout_p11: float = 1.0
) -> NoiseModel:
    error_q1 = depolarizing_error(depolarizing_lambda, 1).compose(
        thermal_relaxation_error(t1, t1, single_q_gate_time)
    )
    error_q2 = depolarizing_error(depolarizing_lambda, 2).compose(
        thermal_relaxation_error(t1, t2, two_q_gate_time).tensor(thermal_relaxation_error(t1, t2, two_q_gate_time))
    )
    error_q3 = depolarizing_error(0.01*depolarizing_lambda, 3).compose(
        thermal_relaxation_error(t1, t2, three_q_gate_time).tensor(thermal_relaxation_error(t1, t2, three_q_gate_time).tensor(thermal_relaxation_error(t1, t2, three_q_gate_time)))
    )
    readout_p_matrix = [[readout_p00, 1-readout_p00], [1-readout_p11, readout_p11]]

    model = NoiseModel(
        basis_gates=["ry", "rx", "id", "x", "rz", "h", "cx", "cswap"]
    )
    model.add_all_qubit_quantum_error(error_q1, ["ry", "rx", "id", "x", "rz", "h"])
    model.add_all_qubit_quantum_error(error_q2, ["cx"])
    model.add_all_qubit_quantum_error(error_q3, ["cswap"])
    model.add_all_qubit_readout_error(ReadoutError(readout_p_matrix))
    return model


# Customized noise model
def noise_model2():
    pass
