from typing import List, Tuple
import math
import QCompute as qc


class QPauliOp(qc.QEnv):
    def __init__(self, num_qubits: int) -> None:
        super().__init__()
        self.q_list = self.Q.createList(num_qubits)
        self._measuredQs = []

    @property
    def measuredQs(self) -> List[int]:
        return self._measuredQs

    def add_x(self, qindex: int) -> None:
        self._measuredQs.append(qindex)
        qc.H(self.q_list[qindex])

    def add_y(self, qindex: int) -> None:
        self._measuredQs.append(qindex)
        qc.RX(math.pi/2)(self.q_list[qindex])

    def add_z(self, qindex: int) -> None:
        self._measuredQs.append(qindex)


class QHamiltonian(object):
    def __init__(self, pauli_strs: List[Tuple[float, str]]) -> None:
        # e.g. pauli_str: [(0.23, "XXIIY"), ....], don't allow [(1.0, "I")]!!!
        self.pauli_strs = pauli_strs

    def h_terms(self) -> List[Tuple[float, QPauliOp]]:
        terms = []
        for coef, pauli in self.pauli_strs:
            qop = QPauliOp(len(pauli))
            for i, op_name in enumerate(pauli):
                if op_name == "X":
                    qop.add_x(i)
                elif op_name == "Y":
                    qop.add_y(i)
                elif op_name == "Z":
                    qop.add_z(i)
            terms.append((coef, qop))
        return terms
