#  ------------------------------------------------------------------------------------------
#  Copyright (c) Microsoft Corporation. All rights reserved.
#  Licensed under the MIT License (MIT). See LICENSE in the repo root for license information.
#  ------------------------------------------------------------------------------------------
import os, sys
import glob
import random
from collections import Counter, OrderedDict
import numpy as np
import torch
import json

import torch
from torch.utils.data import Dataset
from torch.utils.data import DataLoader


class LMOrderedIterator(object):
    def __init__(self, data, bsz, bptt, eval_len=None, device='cpu', world_size=1, rank=0):
        """
            data -- LongTensor -- the LongTensor is strictly ordered
        """
        self.data = data
        self.bsz = bsz
        self.world_size = world_size
        self.rank = rank
        self.bptt = bptt # tgt_len
        # existing len.
        self.eval_len = bptt if eval_len is None else eval_len

        self.device = device
        
        self.global_bsz = bsz * world_size
        # Work out how cleanly we can divide the dataset into bsz parts.
        self.n_step = len(data) // self.global_bsz # bsz

        self.split_data = torch.tensor(
            data[rank * self.n_step * bsz : (rank + 1) * self.n_step * bsz], 
            dtype=torch.long, device=self.device
        )  # data.view(-1)

        self.split_data = self.split_data.view(bsz, -1) 

    def __iter__(self):
        return self.get_fixlen_iter()

    def get_batch(self, i, bptt, eval_len):
        beg_idx = i
        end_idx = i + bptt # seq_len
        
        # batch_size, lengh;
        _input = self.split_data[:, beg_idx : end_idx].contiguous()
        _target = self.split_data[:, beg_idx+1 : end_idx+1].contiguous()

        _msk = torch.cat(
            [
                torch.zeros(bptt-eval_len, dtype=torch.float, device=self.device), 
                torch.ones(eval_len, dtype=torch.float, device=self.device)
            ]
        )
        _msk = _msk.unsqueeze(0).expand_as(_input) # .unsqueeze(-1) # length, 1; 
        return _input, _target, _msk

    def get_fixlen_iter(self, start=0):
        self.data_len = self.split_data.size(1)
        _eval_cursor = 0
        for i in range(start, self.data_len - 1, self.eval_len):
            bptt = min(self.bptt, self.data_len - i - 1)
            _end_idx = i + bptt
            yield self.get_batch(i, bptt, _end_idx - _eval_cursor)
            _eval_cursor = _end_idx 


class Corpus(object):
    def __init__(self, path):
        self.path = path
        self.num_words = 0        
        self.tokens = []
        with open(self.path, "r") as reader:
            for line in reader:
                items = json.loads(line.strip())
                book = items['book']
                tokens = items['tokens']
                num_words = items['num_words']

                self.num_words += num_words
                self.tokens.extend(tokens)


class BinLMOrderedIterator(object):
    def __init__(self, corpus, bsz, bptt, eval_len=None, device='cpu', world_size=1, rank=0):
        """
            data -- LongTensor -- the LongTensor is strictly ordered
        """
        self.corpus = corpus
        self.bsz = bsz
        self.world_size = world_size
        self.rank = rank
        self.bptt = bptt # tgt_len
        # existing len.
        self.eval_len = bptt if eval_len is None else eval_len
        self.device = device
        self.global_bsz = bsz * world_size
        # Work out how cleanly we can divide the dataset into bsz parts.
        self.n_step = corpus.length // self.global_bsz # bsz

        self.offset = [(rank * bsz + _b) * self.n_step  for _b in range(bsz)]

    def __iter__(self):
        return self.get_fixlen_iter()

    def get_batch(self, i, bptt, eval_len):
        # batch_size, lengh;
        _inputs = []
        _targets = []
        for _b in range(0, self.bsz):
            _input = self.corpus.get_tokens(self.offset[_b] + i, bptt)
            _target = self.corpus.get_tokens(self.offset[_b] + i + 1, bptt)

            _inputs.append(_input)
            _targets.append(_target)

        _input = torch.tensor(_inputs, dtype=torch.int64, device=self.device).contiguous()
        _target = torch.tensor(_targets, dtype=torch.int64, device=self.device).contiguous()

        _msk = torch.cat(
            [
                torch.zeros(bptt-eval_len, dtype=torch.float, device=self.device), 
                torch.ones(eval_len, dtype=torch.float, device=self.device)
            ]
        )
        _msk = _msk.unsqueeze(0).expand_as(_input) # .unsqueeze(-1) # length, 1; 
        return _input, _target, _msk

    def get_fixlen_iter(self, start=0):
        #self.data_len = self.split_data.size(1)
        _eval_cursor = 0
        for i in range(start, self.n_step - 1, self.eval_len):
            bptt = min(self.bptt, self.n_step - i - 1)
            _end_idx = i + bptt
            yield self.get_batch(i, bptt, _end_idx - _eval_cursor)
            _eval_cursor = _end_idx 


class BinCorpus(object):
    def __init__(self, path):
        self.path = path

        self.book_token_span = []
        self.book_token_span.append(0)
        tokens_sum = 0
        self.num_words = 0    

        with open(path+'.info', 'r') as info_reader:
            for line in info_reader:
                items = json.loads(line.strip())
                book = items['book']
                num_tokens = items['num_subtokens']
                num_words = items['num_words']

                tokens_sum += num_tokens
                self.book_token_span.append(tokens_sum)
                self.num_words += num_words

        self.length = self.book_token_span[-1]
        self.bin_reader = open(path+'.bin', 'rb')

    def get_tokens(self, offset, count):
        INT64_SIZE = 8
        self.bin_reader.seek(offset * INT64_SIZE)
        x = np.fromfile(self.bin_reader, count=count, dtype=np.int)
        return x


def get_lm_corpus(data):
    print('Producing dataset {}...'.format(data))
    corpus = Corpus(data)
    return corpus


def padding_tokens(tokens, max_seq_length, pad_token, direct, max_context_length=0):

    if max_context_length == 0:
        max_context_length = max_seq_length

    if len(tokens) > max_context_length:
        if direct > 0:
            pad_tokens = tokens[:max_context_length]
        else:
            pad_tokens = tokens[-max_context_length:]
    else:
        pad_tokens = tokens
    token_len = len(pad_tokens)
    pad_tokens = pad_tokens + [pad_token for _ in range(max_seq_length - token_len)]
    return pad_tokens, token_len


class FT_Dataset(Dataset):
    def __init__(self, args, split,
                 max_eval_length=0, prefix_len=0, infix_len=0, 
                 prefix_cursor=1000000, infix_cursor=2000000):
        self.args = args
        self.split = split
        if split == 'eval':
            print('Processing the test data ...')
            self.raw_samples = self.read_ft_file(self.args.test_data)
            self.raw_text = self.load_txt_data(f'./data/{args.data_name}/test_formatted.jsonl')
        else:
            print('Processing the train data ...')
            self.raw_samples = self.read_ft_file(self.args.train_data)
            self.raw_text = self.load_txt_data(f'./data/{args.data_name}/train_formatted.jsonl')
        if args.data_name == 'webnlg_challenge_2017':
            self.full_cate_lst = torch.load(f'./data/{args.data_name}/test_name_dict.pkl') if split == 'eval' else torch.load(f'./data/{args.data_name}/train_name_dict.pkl')
            self.name_indices = self.split_data_by_cate(self.full_cate_lst, split)
        else:
            self.name_indices = self.split_data(self.raw_text)
        if split == 'eval':
            self.ft_samples, self.sample_text = self.construct_het_data(self.name_indices, self.raw_samples, self.raw_text, split)
            self.args.eval_ratio=0
        else:
            self.ft_samples= self.construct_het_data(self.name_indices, self.raw_samples, self.raw_text, split)
        self.batch_size = self.args.train_batch_size
        self.num_examples = len(self.ft_samples)
        self.max_eval_length = max_eval_length
        self.rng = random.Random(911)
        self.joint_lm = args.obj=='jlm'
        self.num_batches = int((self.num_examples + self.batch_size - 1) / self.batch_size)
        self.prefix_len = prefix_len
        self.infix_len = infix_len
        self.prefix_cursor = prefix_cursor
        self.infix_cursor = infix_cursor

    def __len__(self):
        return self.num_batches * self.batch_size
        
    def __getitem__(self, item):
        if(item >= self.num_examples):
            item = self.rng.randint(0, self.num_examples - 1)

        example = self.ft_samples[item]
        context = example[0]
        completion = example[1]

        pretokens = [i + self.prefix_cursor for i in range(0, self.prefix_len)] 
        intokens = [i + self.infix_cursor for i in range(0, self.infix_len)] 

        conditions = pretokens + context + intokens 
        _input, _input_len = padding_tokens(conditions + completion, self.args.seq_len, 0, 1)

        pad_targets = [0 for i in range(0, self.prefix_len)] + context + [0 for i in range(0, self.infix_len)] + completion
        _target, _ = padding_tokens(pad_targets[1:], self.args.seq_len, 0, 1)

        if not self.joint_lm:
            _msk = [0.0] * (len(conditions) - 1) + [1.0] * (_input_len - len(conditions))
        else:
            _msk = [1.0] * (_input_len - 1)

        _msk, _ = padding_tokens(_msk, self.args.seq_len, 0.0, 1)
        
        output = {}
        output["id"] = torch.tensor(item, dtype=torch.long)
        
        _query, _query_len = padding_tokens(
            conditions, self.args.seq_len, 0, -1, 
            max_context_length = self.args.seq_len - self.max_eval_length
        )
        output["query"] = torch.tensor(_query, dtype=torch.long)
        output["query_len"] = torch.tensor(_query_len, dtype=torch.long)

        output["input"] = torch.tensor(_input, dtype=torch.long) 
        output["target"] = torch.tensor(_target, dtype=torch.long) 

        output["mask"] = torch.tensor(_msk, dtype=torch.float)
        return output

    def read_ft_file(self, ft_file):
        ft_samples = []
        with open(ft_file, 'r') as reader:
            for line in reader:
                items = json.loads(line.strip())
                context = items['context']
                completion = items['completion']
                ft_samples.append([context, completion])
        return ft_samples

    def load_txt_data(self, file_path):
        data = []
        with open(file_path, 'r', encoding='utf-8') as f:
            for line in f:
                data.append(json.loads(line.strip()))
                # input_text, output_text = line.strip().split('||')
                # data.append({'input': input_text.strip(), 'output': output_text.strip()})
        return data

    def split_data(self, raw_data):
        name_dict = {}
        for index, entry in enumerate(raw_data):
            context_parts = entry['context'].split(' | ')
            name = None
            for part in context_parts:
                if part.startswith('name : '):
                    name = part.split(' : ')[1]
                    break
            if name:
                if name not in name_dict:
                    name_dict[name] = []
                name_dict[name].append(index)
        return name_dict

    def split_data_by_cate(self, full_cate_list, split):
        categories = ['Airport', 'Astronaut', 'Building', 'City', 'ComicsCharacter', 'Food', 'Monument',
                             'SportsTeam', 'University', 'WrittenWork']
        name_dict = {}
        for i in categories:
            name_dict[i] = []

        for index, cate in enumerate(full_cate_list):
            if cate in categories:
                name_dict[cate].append(index)
        if split == 'eval':
            name_dict['City'] = name_dict['City'][:77]
        return name_dict

    def construct_het_data(self, train_indices, train_data, train_data_text, split):
        train_label_proportions = torch.tensor(
            [float(len(train_indices[i])) for i in train_indices.keys()]
        )
        # names_restaurants = sorted(train_indices.keys())
        if self.args.data_name == 'webnlg_challenge_2017':
            names_restaurants = ['Airport', 'Astronaut', 'Building', 'City', 'ComicsCharacter', 'Food', 'Monument', 'SportsTeam', 'University',
             'WrittenWork']
        else:
            if self.split == 'eval':
                names_restaurants =    ['Blue Spice', 'Clowns', \
                                        'Cocum', 'Cotto',  \
                                        'Giraffe', 'Green Man', \
                                        'Loch Fyne', 'Strada', \
                                        'The Cricketers', 'The Mill', \
                                        'The Phoenix', 'The Plough',\
                                         'The Punter', 'The Vaults', \
                                         'The Waterman', 'The Wrestlers', 'Wildwood', 'Zizzi']
            else:
                names_restaurants = ['Alimentum', 'Aromi', 'Bibimbap House', 'Blue Spice', \
                                     'Browns Cambridge', 'Clowns', 'Cocum', 'Cotto', \
                                     'Fitzbillies', 'Giraffe', 'Green Man', 'Loch Fyne', \
                                     'Midsummer House', 'Strada', 'Taste of Cambridge', 'The Cambridge Blue', \
                                     'The Cricketers', 'The Dumpling Tree', 'The Eagle', 'The Golden Curry', \
                                     'The Golden Palace', 'The Mill', 'The Olive Grove', 'The Phoenix', \
                                     'The Plough', 'The Punter', 'The Rice Boat', 'The Twenty Two', \
                                     'The Vaults', 'The Waterman', 'The Wrestlers', 'Travellers Rest Beefeater',
                                     'Wildwood', 'Zizzi']
        # names_restaurants = train_indices.keys()
        # print(f'=====> The name of restaurants: {names_restaurants}')
        '''
        ['Alimentum', 'Aromi', 'Bibimbap House', 'Blue Spice', 'Browns Cambridge', 'Clowns', 'Cocum', 'Cotto', \
        'Fitzbillies', 'Giraffe', 'Green Man', 'Loch Fyne', 'Midsummer House', 'Strada', 'Taste of Cambridge', 'The Cambridge Blue', \
        'The Cricketers', 'The Dumpling Tree', 'The Eagle', 'The Golden Curry', 'The Golden Palace', 'The Mill', 'The Olive Grove', 'The Phoenix', \
        'The Plough', 'The Punter', 'The Rice Boat', 'The Twenty Two', 'The Vaults', 'The Waterman', 'The Wrestlers', 'Travellers Rest Beefeater', 'Wildwood', 'Zizzi']

        
        ['Blue Spice', 'Clowns', 'Cocum', 'Cotto',  \ 
        'Giraffe', 'Green Man', 'Loch Fyne', 'Strada', \
        'The Cricketers', 'The Mill', 'The Phoenix', 'The Plough',\
         'The Punter', 'The Vaults', 'The Waterman', 'The Wrestlers', 'Wildwood', 'Zizzi']
         
        ['Alimentum', 'Aromi', 'Bibimbap House', 'Blue Spice', \
        'Browns Cambridge', 'Clowns', 'Cocum', 'Cotto', \
        'Fitzbillies', 'Giraffe', 'Green Man', 'Loch Fyne', \
        'Midsummer House', 'Strada', 'Taste of Cambridge', 'The Cambridge Blue', \
        'The Cricketers', 'The Dumpling Tree', 'The Eagle', 'The Golden Curry', \
        'The Golden Palace', 'The Mill', 'The Olive Grove', 'The Phoenix', \
        'The Plough', 'The Punter', 'The Rice Boat', 'The Twenty Two', \
        'The Vaults', 'The Waterman', 'The Wrestlers', 'Travellers Rest Beefeater', 'Wildwood', 'Zizzi']

        
        ['Blue Spice', 'Clowns', \
        'Cocum', 'Cotto',  \ 
        'Giraffe', 'Green Man', \
        'Loch Fyne', 'Strada', \
        'The Cricketers', 'The Mill', \
        'The Phoenix', 'The Plough',\
         'The Punter', 'The Vaults', \
         'The Waterman', 'The Wrestlers', 'Wildwood', 'Zizzi']
        '''
        train_dataset_size = int(torch.sum(train_label_proportions) * (1 - self.args.eval_ratio))
        # eval_dataset_size = torch.sum(train_label_proportions) - train_dataset_size

        for t in names_restaurants:
            random.shuffle(train_indices[t])
            # print(f'{t}: {len(train_indices[t])}')
        # divide samples from each label into iid pool and non-iid pool. Note that samples
        # in iid pool are shuffled while samples in non-iid pool are sorted by label.
        iid_pool = []
        non_iid_pool = []
        for i in names_restaurants:
            iid_split = int((1.0 - self.args.heterogeneity) * len(train_indices[i]))
            iid_pool += train_indices[i][:iid_split]
            non_iid_pool += train_indices[i][iid_split:]
        random.shuffle(iid_pool)
        # Allocate iid and non-iid samples to each worker.
        iid_start = 0
        non_iid_start = 0
        partition_size_train = int(train_dataset_size // self.args.world_size)
        # partition_size_eval = int(eval_dataset_size // self.args.world_size)
        # if self.args.rank == 0:
        #     print(f'The training dataset size: {train_dataset_size}')
        train_worker_idxs = [[] for _ in range(self.args.world_size)]
        train_lower_idxs = [[] for _ in range(self.args.world_size)]
        for j in range(self.args.world_size):
            num_iid_train = int((1.0 - self.args.heterogeneity) * partition_size_train)
            num_non_iid_train = partition_size_train - num_iid_train
            train_worker_idxs[j] += iid_pool[iid_start: iid_start + num_iid_train]
            train_worker_idxs[j] += non_iid_pool[non_iid_start: non_iid_start + num_non_iid_train]
            train_lower_idxs[j] += iid_pool[iid_start: iid_start + num_iid_train]
            train_lower_idxs[j] += non_iid_pool[non_iid_start: non_iid_start + num_non_iid_train]
            iid_start += num_iid_train
            non_iid_start += num_non_iid_train
            random.shuffle(train_worker_idxs[j])
            random.shuffle(train_lower_idxs[j])


        print(f'Rank: {self.args.rank}  dataset size: {len(train_worker_idxs[self.args.rank])}')
        train_dataset = [train_data[idx] for idx in train_worker_idxs[self.args.rank]]
        # train_dataset_text = [train_data_text[idx] for idx in train_worker_idxs[self.args.rank]]
        train_lower = [train_data[idx] for idx in train_lower_idxs[self.args.rank]]
        # train_lower_text = [train_data_text[idx] for idx in train_lower_idxs[self.args.rank]]
        # eval_dataset = [train_data[idx] for idx in eval_worker_idxs[self.args.rank]]
        train_dataset_text = [train_data_text[idx] for idx in train_worker_idxs[self.args.rank]]
        if split == 'train':
            return train_dataset
        elif split == 'eval':
            return train_dataset, train_dataset_text
        else:
           return train_lower



