/***************************************************************************************************
 * Copyright (c) 2017-2020, NVIDIA CORPORATION.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 *modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright notice,
 *this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *notice, this list of conditions and the following disclaimer in the
 *documentation and/or other materials provided with the distribution.
 *     * Neither the name of the NVIDIA CORPORATION nor the names of its
 *contributors may be used to endorse or promote products derived from this
 *software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 *AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *DISCLAIMED. IN NO EVENT SHALL NVIDIA CORPORATION BE LIABLE FOR ANY DIRECT,
 *INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 *OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TOR (INCLUDING
 *NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 *EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************************************/

/*! \file
    \brief
      Default kernel-level GEMM definitions combine threadblock-scoped matrix
   multiply-add with the appropriate threadblock-scoped epilogue.

      Note, CUTLASS epilogues universally target row-major outputs. Column-major
   outputs are accommodated by exchanging A and B operands and assuming
   transposed layouts. Partial specializations here choose
   'device::GemmTransposed' to implement this functionality.

*/

#pragma once

#include "cutlass/cutlass.h"

#include "cutlass/complex.h"
#include "cutlass/layout/matrix.h"
#include "cutlass/numeric_types.h"

#include "cutlass/gemm/kernel/gemm_planar_complex.h"
#include "cutlass/gemm/kernel/gemm_planar_complex_array.h"
#include "cutlass/gemm/kernel/default_gemm.h"
#include "cutlass/gemm/kernel/default_gemm_complex.h"

#include "cutlass/epilogue/threadblock/default_epilogue_planar_complex.h"
#include "cutlass/gemm/threadblock/default_mma_planar_complex_pipelined.h"
#include "cutlass/gemm/threadblock/default_mma_planar_complex_multistage.h"

/////////////////////////////////////////////////////////////////////////////////////////////////

namespace cutlass {
namespace gemm {
namespace kernel {

/////////////////////////////////////////////////////////////////////////////////////////////////

template <
        /// Element type for A matrix operand
        typename ElementA,
        /// Layout type for A matrix operand
        typename LayoutA,
        /// Complex elementwise transformation on A operand
        ComplexTransform TransformA,
        /// Access granularity of A matrix in units of elements
        int kAlignmentA,
        /// Element type for B matrix operand
        typename ElementB,
        /// Layout type for B matrix operand
        typename LayoutB,
        /// Complex elementwise transformation on B operand
        ComplexTransform TransformB,
        /// Access granularity of B matrix in units of elements
        int kAlignmentB,
        /// Element type for C and D matrix operands
        typename ElementC,
        /// Layout type for C and D matrix operands
        typename LayoutC,
        /// Element type for internal accumulation
        typename ElementAccumulator,
        /// Operator class tag
        typename OperatorClass,
        /// Tag indicating architecture to tune for
        typename ArchTag,
        /// Threadblock-level tile size (concept: GemmShape)
        typename ThreadblockShape,
        /// Warp-level tile size (concept: GemmShape)
        typename WarpShape,
        /// Warp-level tile size (concept: GemmShape)
        typename InstructionShape,
        /// Epilogue output operator
        typename EpilogueOutputOp,
        /// Threadblock-level swizzling operator
        typename ThreadblockSwizzle,
        /// Number of stages used in the pipelined mainloop
        int Stages,
        /// Math operation performed by GEMM (e.g. arch::OpMultiplyAdd)
        typename Operator,
        /// Conditional enabling to switch between stages
        typename Enable = void>
struct DefaultGemmPlanarComplexUniversal;

/////////////////////////////////////////////////////////////////////////////////////////////////

/// Partial specialization for pipelined mainloop
template <
        /// Element type for A matrix operand
        typename ElementA,
        /// Layout type for A matrix operand
        typename LayoutA,
        /// Complex elementwise transformation on A operand
        ComplexTransform TransformA,
        /// Access granularity of A matrix in units of elements
        int kAlignmentA,
        /// Element type for B matrix operand
        typename ElementB,
        /// Layout type for B matrix operand
        typename LayoutB,
        /// Complex elementwise transformation on B operand
        ComplexTransform TransformB,
        /// Access granularity of B matrix in units of elements
        int kAlignmentB,
        /// Element type for C and D matrix operands
        typename ElementC,
        /// Layout type for C and D matrix operands
        typename LayoutC,
        /// Element type for internal accumulation
        typename ElementAccumulator,
        /// Operator class tag
        typename OperatorClass,
        /// Tag indicating architecture to tune for
        typename ArchTag,
        /// Threadblock-level tile size (concept: GemmShape)
        typename ThreadblockShape,
        /// Warp-level tile size (concept: GemmShape)
        typename WarpShape,
        /// Warp-level tile size (concept: GemmShape)
        typename InstructionShape,
        /// Epilogue output operator
        typename EpilogueOutputOp,
        /// Threadblock-level swizzling operator
        typename ThreadblockSwizzle,
        /// Number of stages used in the pipelined mainloop
        int Stages,
        /// Operation performed by GEMM
        typename Operator>
struct DefaultGemmPlanarComplexUniversal<
        ElementA, LayoutA, TransformA, kAlignmentA, ElementB, LayoutB,
        TransformB, kAlignmentB, ElementC, LayoutC, ElementAccumulator,
        OperatorClass, ArchTag, ThreadblockShape, WarpShape, InstructionShape,
        EpilogueOutputOp, ThreadblockSwizzle, Stages, Operator,
        typename std::enable_if<(Stages <= 2)>::type> {
    /// Define planar complex valued variants instead
    using Mma = typename gemm::threadblock::DefaultMmaPlanarComplexPipelined<
            ElementA, LayoutA, kAlignmentA, ElementB, LayoutB, kAlignmentB,
            ElementAccumulator, LayoutC, OperatorClass, ArchTag,
            ThreadblockShape, WarpShape, InstructionShape, Stages, TransformA,
            TransformB, Operator>::ThreadblockMma;

    /// Planar complex epilogue
    using Epilogue =
            typename epilogue::threadblock::DefaultEpiloguePlanarComplex<
                    ThreadblockShape, typename Mma::Policy::Operator,
                    OperatorClass, ArchTag,
                    ThreadblockShape::kK / WarpShape::kK, EpilogueOutputOp,
                    EpilogueOutputOp::kCount>::Epilogue;

    /// Define the kernel in terms of the default kernel
    using GemmKernel =
            kernel::GemmPlanarComplex<Mma, Epilogue, ThreadblockSwizzle>;

    // Array variant
    using GemmArrayKernel =
            kernel::GemmPlanarComplexArray<Mma, Epilogue, ThreadblockSwizzle>;
};

/////////////////////////////////////////////////////////////////////////////////////////////////

/// Partial specialization for multiple pipeline stages.
template <
        /// Element type for A matrix operand
        typename ElementA,
        /// Layout type for A matrix operand
        typename LayoutA,
        /// Complex elementwise transformation on A operand
        ComplexTransform TransformA,
        /// Access granularity of A matrix in units of elements
        int kAlignmentA,
        /// Element type for B matrix operand
        typename ElementB,
        /// Layout type for B matrix operand
        typename LayoutB,
        /// Complex elementwise transformation on B operand
        ComplexTransform TransformB,
        /// Access granularity of B matrix in units of elements
        int kAlignmentB,
        /// Element type for C and D matrix operands
        typename ElementC,
        /// Layout type for C and D matrix operands
        typename LayoutC,
        /// Element type for internal accumulation
        typename ElementAccumulator,
        /// Operator class tag
        typename OperatorClass,
        /// Tag indicating architecture to tune for
        typename ArchTag,
        /// Threadblock-level tile size (concept: GemmShape)
        typename ThreadblockShape,
        /// Warp-level tile size (concept: GemmShape)
        typename WarpShape,
        /// Warp-level tile size (concept: GemmShape)
        typename InstructionShape,
        /// Epilogue output operator
        typename EpilogueOutputOp,
        /// Threadblock-level swizzling operator
        typename ThreadblockSwizzle,
        /// Number of stages used in the pipelined mainloop
        int Stages,
        /// Operation performed by GEMM
        typename Operator>
struct DefaultGemmPlanarComplexUniversal<
        ElementA, LayoutA, TransformA, kAlignmentA, ElementB, LayoutB,
        TransformB, kAlignmentB, ElementC, LayoutC, ElementAccumulator,
        OperatorClass, ArchTag, ThreadblockShape, WarpShape, InstructionShape,
        EpilogueOutputOp, ThreadblockSwizzle, Stages, Operator,
        typename std::enable_if<(Stages > 2)>::type> {
    /// Define planar complex valued variants instead
    using Mma = typename gemm::threadblock::DefaultMmaPlanarComplexMultistage<
            ElementA, LayoutA, kAlignmentA, ElementB, LayoutB, kAlignmentB,
            ElementAccumulator, LayoutC, OperatorClass, ArchTag,
            ThreadblockShape, WarpShape, InstructionShape, Stages, TransformA,
            TransformB, Operator>::ThreadblockMma;

    /// Planar complex epilogue
    using Epilogue =
            typename epilogue::threadblock::DefaultEpiloguePlanarComplex<
                    ThreadblockShape, typename Mma::Policy::Operator,
                    OperatorClass, ArchTag,
                    ThreadblockShape::kK / WarpShape::kK, EpilogueOutputOp,
                    EpilogueOutputOp::kCount>::Epilogue;

    /// Define the kernel in terms of the default kernel
    using GemmKernel =
            kernel::GemmPlanarComplex<Mma, Epilogue, ThreadblockSwizzle>;

    // Array variant
    using GemmArrayKernel =
            kernel::GemmPlanarComplexArray<Mma, Epilogue, ThreadblockSwizzle>;
};

/////////////////////////////////////////////////////////////////////////////////////////////////

}  // namespace kernel
}  // namespace gemm
}  // namespace cutlass

/////////////////////////////////////////////////////////////////////////////////////////////////
