import os
import os.path as osp
import pathlib
from typing import Any, Sequence

import torch
import torch.nn.functional as F
from rdkit import Chem, RDLogger
from tqdm import tqdm
import numpy as np
import pandas as pd
from torch_geometric.data import Data, InMemoryDataset, download_url, extract_zip
from hydra.utils import get_original_cwd

from didigress.datasets.dataset_utils import mol_to_torch_geometric, remove_hydrogens, StatisticsMolecule
from didigress.datasets.dataset_utils import load_pickle, save_pickle
from didigress.datasets.abstract_dataset import AbstractDatasetInfos, AbstractDataModule, AbstractAdaptiveDataModule
from didigress.metrics.metrics_utils import compute_all_statistics, atom_type_counts, edge_counts, charge_counts
from didigress.utils import PlaceHolder

HAR2EV = 27.211386246
KCALMOL2EV = 0.04336414

conversion = torch.tensor([1., 1., HAR2EV, HAR2EV, HAR2EV, 1., HAR2EV, HAR2EV, HAR2EV, HAR2EV, HAR2EV,
                           1., KCALMOL2EV, KCALMOL2EV, KCALMOL2EV, KCALMOL2EV, 1., 1., 1.])

def files_exist(files) -> bool:
    # NOTE: We return `False` in case `files` is empty, leading to a
    # re-processing of files on every instantiation.
    return len(files) != 0 and all([osp.exists(f) for f in files])


def to_list(value: Any) -> Sequence:
    if isinstance(value, Sequence) and not isinstance(value, str):
        return value
    else:
        return [value]
    
class RemoveYTransform:
    def __call__(self, data):
        data.guidance = torch.zeros((1, 0), dtype=torch.float)
        return data
    
class FeatureExtractorTransform:
    def __init__(self, guidance_target):
        self.guidance_target = guidance_target

    def __call__(self, data):
        guidance = []
        if 'mu' in self.guidance_target:
            guidance.append(data.guidance[..., :1])
        if 'homo' in self.guidance_target:
            guidance.append(data.guidance[..., 1:])
        
        data.guidance = torch.hstack(tuple(guidance))

        return data


full_atom_encoder = {'H': 0, 'C': 1, 'N': 2, 'O': 3, 'F': 4}

#TODO: remove pos and charge when not needed?
class QM9Dataset(InMemoryDataset):
    raw_url = ('https://deepchemdata.s3-us-west-1.amazonaws.com/datasets/'
               'molnet_publish/qm9.zip')
    raw_url2 = 'https://ndownloader.figshare.com/files/3195404'
    processed_url = 'https://data.pyg.org/datasets/qm9_v3.zip'

    def __init__(self, split, root, remove_h: bool, cfg, transform=None, pre_transform=None, pre_filter=None):
        self.split = split
        if self.split == 'train':
            self.file_idx = 0
        elif self.split == 'val':
            self.file_idx = 1
        else:
            self.file_idx = 2
        self.remove_h = remove_h

        self.cfg = cfg
        self.use_charges = self.cfg.features.use_charges
        self.use_3d = self.cfg.features.use_3d

        self.atom_encoder = full_atom_encoder
        if remove_h:
            self.atom_encoder = {k: v - 1 for k, v in self.atom_encoder.items() if k != 'H'}

        super().__init__(root, transform, pre_transform, pre_filter)
        self.data, self.slices = torch.load(self.processed_paths[0])


        valencies=load_pickle(self.processed_paths[5])

        charge_types=torch.from_numpy(np.load(self.processed_paths[4], allow_pickle=True)).float()
        bond_lengths=load_pickle(self.processed_paths[6])
        bond_angles=torch.from_numpy(np.load(self.processed_paths[7], allow_pickle=True)).float()
        
        self.statistics = StatisticsMolecule(num_nodes=load_pickle(self.processed_paths[1]),
                                     node_types=torch.from_numpy(np.load(self.processed_paths[2])).float(),
                                     edge_types=torch.from_numpy(np.load(self.processed_paths[3])).float(),
                                     cfg=self.cfg,
                                     charge_types=charge_types,
                                     valencies=valencies,
                                     bond_lengths=bond_lengths,
                                     bond_angles=bond_angles)
        self.smiles = load_pickle(self.processed_paths[8])

    @property
    def raw_file_names(self):
        return ['gdb9.sdf', 'gdb9.sdf.csv', 'uncharacterized.txt']

    @property
    def split_file_name(self):
        return ['train.csv', 'val.csv', 'test.csv']

    @property
    def split_paths(self):
        r"""The absolute filepaths that must be present in order to skip
        splitting."""
        files = to_list(self.split_file_name)
        return [osp.join(self.raw_dir, f) for f in files]

    @property
    def processed_file_names(self):
        #TODO:  do not change the return arrays. they are used below (self.processed_paths)
        #       and it's better if we leave them untouched.
        h = 'noh' if self.remove_h else 'h'
        if self.split == 'train':
            return [f'train_{h}.pt', f'train_n_{h}.pickle', f'train_node_types_{h}.npy', f'train_edge_types_{h}.npy',
                    f'train_charges_{h}.npy', f'train_valency_{h}.pickle', f'train_bond_lengths_{h}.pickle',
                    f'train_angles_{h}.npy', 'train_smiles.pickle']
        elif self.split == 'val':
            return [f'val_{h}.pt', f'val_n_{h}.pickle', f'val_node_types_{h}.npy', f'val_edge_types_{h}.npy',
                    f'val_charges_{h}.npy', f'val_valency_{h}.pickle', f'val_bond_lengths_{h}.pickle',
                    f'val_angles_{h}.npy', 'val_smiles.pickle']
        else:
            return [f'test_{h}.pt', f'test_n_{h}.pickle', f'test_node_types_{h}.npy', f'test_edge_types_{h}.npy',
                    f'test_charges_{h}.npy', f'test_valency_{h}.pickle', f'test_bond_lengths_{h}.pickle',
                    f'test_angles_{h}.npy', 'test_smiles.pickle']

    def download(self):
        """
        Download raw qm9 files. Taken from PyG QM9 class
        """
        try:
            import rdkit  # noqa
            file_path = download_url(self.raw_url, self.raw_dir)
            extract_zip(file_path, self.raw_dir)
            os.unlink(file_path)

            file_path = download_url(self.raw_url2, self.raw_dir)
            os.rename(osp.join(self.raw_dir, '3195404'),
                      osp.join(self.raw_dir, 'uncharacterized.txt'))
        except ImportError:
            path = download_url(self.processed_url, self.raw_dir)
            extract_zip(path, self.raw_dir)
            os.unlink(path)

        if files_exist(self.split_paths):
            return

        dataset = pd.read_csv(self.raw_paths[1])

        n_samples = len(dataset)
        n_train = 100000
        n_test = int(0.1 * n_samples)
        n_val = n_samples - (n_train + n_test)

        # Shuffle dataset with df.sample, then split
        train, val, test = np.split(dataset.sample(frac=1, random_state=42), [n_train, n_val + n_train])

        train.to_csv(os.path.join(self.raw_dir, 'train.csv'))
        val.to_csv(os.path.join(self.raw_dir, 'val.csv'))
        test.to_csv(os.path.join(self.raw_dir, 'test.csv'))

    def process(self):
        RDLogger.DisableLog('rdApp.*')

        target_df = pd.read_csv(self.split_paths[self.file_idx], index_col=0)
        target_df.drop(columns=['mol_id'], inplace=True)

        target = torch.tensor(target_df.values, dtype=torch.float)
        target = torch.cat([target[:, 3:], target[:, :3]], dim=-1)
        target = target * conversion.view(1, -1)

        with open(self.raw_paths[-1], 'r') as f:
            skip = [int(x.split()[0]) - 1 for x in f.read().split('\n')[9:-2]]

        suppl = Chem.SDMolSupplier(self.raw_paths[0], removeHs=False, sanitize=False)
        data_list = []
        all_smiles = []
        num_errors = 0
        for i, mol in enumerate(tqdm(suppl)):
            if i in skip or i not in target_df.index:
                continue
            smiles = Chem.MolToSmiles(mol, isomericSmiles=False)
            if smiles is None:
                num_errors += 1
            else:
                all_smiles.append(smiles)

            data = mol_to_torch_geometric(mol=mol, atom_encoder=full_atom_encoder, smiles=smiles, cfg=self.cfg)
            if self.remove_h:
                data = remove_hydrogens(data=data,cfg=self.cfg)

            if self.pre_filter is not None and not self.pre_filter(data):
                continue
            if self.pre_transform is not None:
                data = self.pre_transform(data)
                
            node_stats   = atom_type_counts(data_list=[data], num_classes=len(self.atom_encoder), verbose=False)
            edge_stats   = edge_counts(data_list=[data], verbose=False)
            if(np.isnan(edge_stats).all()):
                print("found a NaN edge tensor")
                edge_stats = np.zeros_like(edge_stats)
                edge_stats[0] = 1 
            charge_types = charge_counts(data_list=[data], num_classes=len(self.atom_encoder), 
                                         charges_dic={-1: 0, 0: 1, 1: 2}, verbose=False)
            
            data.node_stats = torch.from_numpy(node_stats).unsqueeze(0)
            data.edge_stats = torch.from_numpy(edge_stats).unsqueeze(0)
            data.charge_types = torch.from_numpy(charge_types).unsqueeze(0)

            guidance = target[target_df.index.get_loc(i)].unsqueeze(0)
            guidance = torch.hstack((guidance[..., :1], guidance[..., 2:3]))
            data.guidance = guidance

            # Useful for ins_del operations
            data.n_nodes = data.x.size(0)

            data_list.append(data)
        torch.save(self.collate(data_list), self.processed_paths[self.file_idx])

        statistics = compute_all_statistics(data_list, self.atom_encoder, charges_dic={-1: 0, 0: 1, 1: 2}, cfg=self.cfg)

        save_pickle(statistics.num_nodes, self.processed_paths[1])
        np.save(self.processed_paths[2], statistics.node_types)
        np.save(self.processed_paths[3], statistics.edge_types)
        np.save(self.processed_paths[4], statistics.charge_types)
        save_pickle(statistics.valencies, self.processed_paths[5])
        save_pickle(statistics.bond_lengths, self.processed_paths[6])
        np.save(self.processed_paths[7], statistics.bond_angles)
        print("Number of molecules that could not be mapped to smiles: ", num_errors)
        save_pickle(set(all_smiles), self.processed_paths[8])
        torch.save(self.collate(data_list), self.processed_paths[0])


class QM9DataModule(AbstractDataModule):
    def __init__(self, cfg):
        self.datadir = cfg.dataset.datadir
        base_path = pathlib.Path(get_original_cwd()).parents[0]
        root_path = os.path.join(base_path, self.datadir)

        target = cfg.guidance.guidance_target
        if len(set(['mu', 'homo']).intersection(target)) > 0:
            transform = FeatureExtractorTransform(cfg.guidance.guidance_target)
        else:
            transform = RemoveYTransform()

        train_dataset = QM9Dataset(split='train', root=root_path, remove_h=cfg.dataset.remove_h, cfg=cfg, transform=transform)
        val_dataset = QM9Dataset(split='val', root=root_path, remove_h=cfg.dataset.remove_h, cfg=cfg, transform=transform)
        test_dataset = QM9Dataset(split='test', root=root_path, remove_h=cfg.dataset.remove_h, cfg=cfg, transform=transform)
        self.statistics = {'train': train_dataset.statistics, 'val': val_dataset.statistics,
                           'test': test_dataset.statistics}
        self.remove_h = cfg.dataset.remove_h
        super().__init__(cfg, train_dataset=train_dataset, val_dataset=val_dataset, test_dataset=test_dataset)


class QM9infos(AbstractDatasetInfos):
    def __init__(self, datamodule, cfg):
        self.remove_h = cfg.dataset.remove_h
        self.statistics = datamodule.statistics
        self.name = 'qm9'
        self.atom_encoder = full_atom_encoder
        self.collapse_charges = torch.Tensor([-1, 0, 1]).int()
        if self.remove_h:
            self.atom_encoder = {k: v - 1 for k, v in self.atom_encoder.items() if k != 'H'}
        super().complete_infos(datamodule.statistics, self.atom_encoder, cfg)
        
        use_conditional = cfg.guidance.p_uncond >= 0
        medium_extra = {'X': 0, 'E': 0, 'y': 0, 'p': 0,}
        guidance_sizes = {'homo': 1, 'mu': 1}
        self.guidance_dims = 0
        if(use_conditional):
            guidance_mediums = cfg.guidance.guidance_medium
            targets = cfg.guidance.guidance_target

            for target in targets:
                self.guidance_dims += guidance_sizes[target]
                for medium in guidance_mediums:
                    medium_extra[medium] = medium_extra[medium] + guidance_sizes[target]

        print("self.guidance_dims =", self.guidance_dims)
        print("medium_extra =", medium_extra)

        self.use_ins_del= cfg.features.use_ins_del
        IN_ins_del_sz = 2 if self.use_ins_del else 0

        X_sz_out = self.num_node_types + IN_ins_del_sz
        E_sz_out = 5 + IN_ins_del_sz
        y_sz_out = 0
        c_sz_out = 3 + IN_ins_del_sz
        p_sz_out = 3 + IN_ins_del_sz

        X_sz = X_sz_out + medium_extra['X']
        E_sz = E_sz_out + medium_extra['E']
        y_sz = y_sz_out + medium_extra['y'] + 1 #The "+1" is the timestep t
        c_sz = c_sz_out 
        p_sz = p_sz_out + medium_extra['p']


        #TODO: add INS/DEL categories
        self.input_dims = PlaceHolder(X=X_sz, charges=c_sz, E=E_sz, y=y_sz, pos=p_sz, guidance=self.guidance_dims)
        self.output_dims = PlaceHolder(X=X_sz_out, charges=c_sz_out, E=E_sz_out, y=0, pos=p_sz_out, guidance=0)

        self.cfg = cfg
        self.charges_dic = {-1: 0, 0: 1, 1: 2}

    def to_one_hot(self, X,  E, node_mask, charges = None):
        X = F.one_hot(X, num_classes=self.num_node_types).float()
        E = F.one_hot(E, num_classes=5).float()

        if(self.cfg.features.use_charges):
            charges = F.one_hot(charges + 1, num_classes=3).float()
        
        placeholder = PlaceHolder(X=X, E=E, y=None, charges=charges, pos=None)
        pl = placeholder.mask(node_mask)
        return pl.X, pl.E, pl.charges

    def one_hot_charges(self, charges):
        return F.one_hot((charges + 1).long(), num_classes=3).float()
