from typing import Union

import numpy as np


class Feature:
    def __init__(self, d: int, s: int, mode: str):
        '''
        param d: dimension of the feature vector
        param s: number of states
        param mode: type of feature
        '''
        assert mode in ['uniform', 'gaussian', 'one-hot'], "Unknown mode"
        self.d = d
        self.s = s
        if mode == 'uniform':
            self.phi = np.random.uniform(low=-1, high=1,
                                         size=(s, d)).astype(np.float32)
        elif mode == 'gaussian':
            self.phi = np.random.randn(s, d).astype(np.float32)
        elif mode == 'one-hot':
            assert s == d, "number of states must be equal to the feature dimension"
            self.phi = np.eye(s, dtype=np.float32)

    def __call__(self, s: Union[int, np.ndarray]) -> np.ndarray:
        '''
        param s: state or array of states
        '''
        return self.phi[s]

    def copy(self) -> 'Feature':
        f = Feature(self.d, self.s)
        f.phi = self.phi.copy()
        return f
