import random

import numpy as np
import scipy as sp
import torch


def compute_steady_dist(P: np.array) -> np.ndarray:
    '''
    param P: transition probability matrix
    '''

    n = P.shape[0]
    null_vec = sp.linalg.null_space(np.eye(n) - P.T)

    return (null_vec / np.sum(null_vec)).flatten()


def compute_msve(v_hat: np.ndarray,
                 v: np.ndarray,
                 steady_dist: np.ndarray) -> float:
    '''
    param v_hat: predicted value
    param v: true value
    param steady_dist: steady state distribution
    returns MSVE between v_hat and v
    '''
    error = v - v_hat
    msve = steady_dist.dot(error**2)
    return msve.item()


def scale(matrix: np.ndarray) -> np.ndarray:
    '''
    param matrix: input matrix
    scale the matrix to the range [-1, 1]
    '''
    return matrix / np.max(np.abs(matrix))


def set_seed(seed: int):
    '''
    param seed: int (random seed)
    set the random seed for numpy, torch and random
    '''
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)

def make_exp_name(cfg: dict) -> str:
    '''
    param cfg: dict (configuration dictionary)
    returns: str (experiment name)
    '''
    name = cfg["mrp"]
    name += f'_{cfg["feature_mode"]}'
    name += '_repr' if cfg['representable'] else '_non-repr'
    name += '_enumerated_ctxt' if cfg['enumerated_context'] else '_sampled_ctxt'
    name += '_expected_feat' if cfg['expected_feature'] else '_sampled_feat'
    name += '_linear' if cfg['activation'] == 'identity' else f'_{cfg["activation"]}'
    name += f'_{cfg["target"]}'
    return name
