import torch
import torch.nn as nn
import torch.nn.functional as F
import timm
from timm.models.layers import DropPath, trunc_normal_
import numpy as np
from .build import MODELS
from utils import misc
from utils.checkpoint import get_missing_parameters_message, get_unexpected_parameters_message
from utils.logger import *
import random
from knn_cuda import KNN
from extensions.chamfer_dist import ChamferDistanceL1, ChamferDistanceL2


class Encoder(nn.Module):   ## Embedding module
    def __init__(self, encoder_channel):
        super().__init__()
        self.encoder_channel = encoder_channel
        self.first_conv = nn.Sequential(
            nn.Conv1d(3, 128, 1),
            nn.BatchNorm1d(128),
            nn.ReLU(inplace=True),
            nn.Conv1d(128, 256, 1)
        )
        self.second_conv = nn.Sequential(
            nn.Conv1d(512, 512, 1),
            nn.BatchNorm1d(512),
            nn.ReLU(inplace=True),
            nn.Conv1d(512, self.encoder_channel, 1)
        )

    def forward(self, point_groups):
        '''
            point_groups : B G N 3
            -----------------
            feature_global : B G C
        '''
        bs, g, n , _ = point_groups.shape
        point_groups = point_groups.reshape(bs * g, n, 3)
        # encoder
        feature = self.first_conv(point_groups.transpose(2,1))  # BG 256 n
        feature_global = torch.max(feature,dim=2,keepdim=True)[0]  # BG 256 1
        feature = torch.cat([feature_global.expand(-1,-1,n), feature], dim=1)# BG 512 n
        feature = self.second_conv(feature) # BG 1024 n
        feature_global = torch.max(feature, dim=2, keepdim=False)[0] # BG 1024
        return feature_global.reshape(bs, g, self.encoder_channel)


class Group(nn.Module):  # FPS + KNN
    def __init__(self, num_group, group_size):
        super().__init__()
        self.num_group = num_group
        self.group_size = group_size
        self.knn = KNN(k=self.group_size, transpose_mode=True)

    def forward(self, xyz):
        '''
            input: B N 3
            ---------------------------
            output: B G M 3
            center : B G 3
        '''
        batch_size, num_points, _ = xyz.shape
        # fps the centers out
        center = misc.fps(xyz, self.num_group) # B G 3
        # knn to get the neighborhood
        _, idx = self.knn(xyz, center) # B G M
        assert idx.size(1) == self.num_group
        assert idx.size(2) == self.group_size
        idx_base = torch.arange(0, batch_size, device=xyz.device).view(-1, 1, 1) * num_points
        idx = idx + idx_base
        idx = idx.view(-1)
        neighborhood = xyz.view(batch_size * num_points, -1)[idx, :]
        neighborhood = neighborhood.view(batch_size, self.num_group, self.group_size, 3).contiguous()
        # normalize
        neighborhood = neighborhood - center.unsqueeze(2)
        return neighborhood, center


## Transformers
class Mlp(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, act_layer=nn.GELU, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features or in_features
        self.fc1 = nn.Linear(in_features, hidden_features)
        self.act = act_layer()
        self.fc2 = nn.Linear(hidden_features, out_features)
        self.drop = nn.Dropout(drop)

    def forward(self, x):
        x = self.fc1(x)
        x = self.act(x)
        x = self.drop(x)
        x = self.fc2(x)
        x = self.drop(x)
        return x


class Attention(nn.Module):
    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        # NOTE scale factor was wrong in my original version, can set manually to be compat with prev weights
        self.scale = qk_scale or head_dim ** -0.5
        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)
    def score_assignment_step(self, attn, v):
        """
        Token Score Assignment Step.
        :param attn: attention matrix
        :param v: values
        :return: sorted significance scores and their corresponding indices
        """
        B, H, _, _ = attn.shape
        C = v.shape[3] * H
        v_norm = torch.linalg.norm(
            v.transpose(1, 2).reshape(B, attn.shape[2], C), ord=2, dim=2
        )  # value norm of size [B x T]
        significance_score = attn[:, :, 0].sum(
            dim=1
        )  # attention weights of CLS token of size [B x T]
        significance_score = significance_score * v_norm  # [B x T]
        significance_score = significance_score[:, 1:]  # [B x T-1]


        return significance_score

    def sampling(self, significance_score, mask_ratio):
        """
        Sample tokens based on their significance scores.
        """
        B, K = significance_score.shape
        
        mask_num = int( (1-mask_ratio)*K )

        temperature = 0.5
        probabilities = torch.softmax((significance_score/ temperature), dim=1)
        beta = torch.rand(B, K).to(significance_score.device)
        r = -torch.log(-torch.log(beta))
        probabilities = torch.log(probabilities)+r

        sorted_patch, sorted_indices = torch.sort(probabilities, descending=False, dim=1)

        tokens_to_pick_ind = sorted_indices[:, mask_num:] #[B, M]
        mask = torch.zeros_like(significance_score, dtype=torch.bool)
        mask[torch.arange(B).view(-1, 1), tokens_to_pick_ind] = True

        return mask

    def forward(self, x, mask_ratio, last_block):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]   # make torchscript happy (cannot use tensor as tuple)

        attn = (q @ k.transpose(-2, -1)) * self.scale
        attn = attn.softmax(dim=-1)

        if last_block:
            score = self.score_assignment_step(attn, v)
            mask = self.sampling(score, mask_ratio)
        else:
            mask = None
            score = None

        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x, mask, score


class Block(nn.Module):
    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm):
        super().__init__()
        self.norm1 = norm_layer(dim)

        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

        self.attn = Attention(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        
    def forward(self, x, mask_ratio, last_block):
        new_x, mask, score = self.attn(self.norm1(x), mask_ratio, last_block)
        x = x + self.drop_path(new_x)
        x = x + self.drop_path(self.mlp(self.norm2(x)))
        return x, mask, score


class TransformerEncoder(nn.Module):
    def __init__(self, embed_dim=768, depth=4, num_heads=12, mlp_ratio=4., qkv_bias=False, qk_scale=None,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0.):
        super().__init__()
        self.depth = depth
        self.blocks = nn.ModuleList([
            Block(
                dim=embed_dim, num_heads=num_heads, mlp_ratio=mlp_ratio, qkv_bias=qkv_bias, qk_scale=qk_scale,
                drop=drop_rate, attn_drop=attn_drop_rate, 
                drop_path = drop_path_rate[i] if isinstance(drop_path_rate, list) else drop_path_rate
                )
            for i in range(depth)])

    def forward(self, x, pos, mask_ratio):
        last_block = False
        for idx, block in enumerate(self.blocks):
            if idx == self.depth-1 :
                last_block = True
            x = x + pos
            x, mask, score = block(x, mask_ratio, last_block)
        return x, mask, score


# finetune model
@MODELS.register_module()
class PointTransformer(nn.Module):
    def __init__(self, config, **kwargs):
        super().__init__()
        self.config = config

        self.trans_dim = config.trans_dim
        self.depth = config.depth
        self.drop_path_rate = config.drop_path_rate
        self.cls_dim = config.cls_dim
        self.num_heads = config.num_heads

        self.group_size = config.group_size
        self.num_group = config.num_group
        self.encoder_dims = config.encoder_dims

        self.group_divider = Group(num_group=self.num_group, group_size=self.group_size)

        self.encoder = Encoder(encoder_channel=self.encoder_dims)

        self.cls_token = nn.Parameter(torch.zeros(1, 1, self.trans_dim))
        self.cls_pos = nn.Parameter(torch.randn(1, 1, self.trans_dim))
        
        self.mask_ratio = config.mask_ratio
        #self.attn_mask_token = nn.Parameter(torch.zeros(1, 1, self.trans_dim))
        self.mask_token = nn.Parameter(torch.zeros(1, 1, self.trans_dim))
        #self.mask_pos = nn.Parameter(torch.randn(1, 1, self.trans_dim))

        self.pos_embed = nn.Sequential(
            nn.Linear(3, 128),
            nn.GELU(),
            nn.Linear(128, self.trans_dim)
        )

        dpr = [x.item() for x in torch.linspace(0, self.drop_path_rate, self.depth)]
        self.blocks = TransformerEncoder(
            embed_dim=self.trans_dim,
            depth=self.depth,
            drop_path_rate=dpr,
            num_heads=self.num_heads,
        )

        self.norm = nn.LayerNorm(self.trans_dim)

        self.cls_head_finetune = nn.Sequential(
                nn.Linear(self.trans_dim * 2, 256),
                nn.BatchNorm1d(256),
                nn.ReLU(inplace=True),
                nn.Dropout(0.5),
                nn.Linear(256, 256),
                nn.BatchNorm1d(256),
                nn.ReLU(inplace=True),
                nn.Dropout(0.5),
                nn.Linear(256, self.cls_dim)
            )

        self.build_loss_func()
        
        self.use_sample_contras_loss = config.use_sample_contras_loss
        self.use_class_contras_loss = config.use_class_contras_loss

        trunc_normal_(self.cls_token, std=.02)
        trunc_normal_(self.cls_pos, std=.02)
        #trunc_normal_(self.attn_mask_token, std=.02)
        trunc_normal_(self.mask_token, std=.02)
        #trunc_normal_(self.mask_pos, std=.02)

    def build_loss_func(self):
        self.loss_ce = nn.CrossEntropyLoss()

    def get_standard_loss_acc(self, ret, gt):
        loss = self.loss_ce(ret, gt.long())
        pred = ret.argmax(-1)
        acc = (pred == gt).sum() / float(gt.size(0))
        return loss, acc * 100    
    
    def class_contras_loss(self, feat1, feat2, feature_class, gt, logit_scale=0.7):
        logits_S = (F.normalize(feat1, dim=1) @ F.normalize(feature_class, dim=1).T) / logit_scale
        logits_M = (F.normalize(feat2, dim=1) @ F.normalize(feature_class, dim=1).T) / logit_scale
        loss = (F.cross_entropy(logits_S, gt.long()) + F.cross_entropy(logits_M, gt.long()))/ 2
        return loss
        
    def contras_loss(self, feat1, feat2, gt, logit_scale=0.7):
        logits_S = (F.normalize(feat1, dim=1) @ F.normalize(feat1, dim=1).T) / logit_scale
        logits_M = (F.normalize(feat1, dim=1) @ F.normalize(feat2, dim=1).T) / logit_scale
        mask_SM = torch.eye(logits_S.shape[0], dtype=torch.bool).to(logits_S.device)
        logits = torch.where(mask_SM, logits_M, logits_S)
        
        labels = torch.arange(logits.shape[0]).to(logits.device)

        # Create mask to filter out same class samples
        gt = gt.view(-1, 1)  # (batch_size, 1)
        mask = (gt == gt.T).to(logits.device)  # (batch_size, batch_size)
        mask.fill_diagonal_(False)
        logits = logits.masked_fill(mask, float('-1e9'))
        
        loss = (F.cross_entropy(logits, labels) + F.cross_entropy(logits.T, labels)) / 2
        return loss

    def get_loss_acc(self, feature_s, feature_m, feature_class, ret_s, ret_m, gt, use_sample_loss):
        
        pred_loss_s = self.loss_ce(ret_s, gt.long())
        loss = pred_loss_s
        if ret_m != None:
            pred_loss_m = self.loss_ce(ret_m, gt.long())
            loss = loss + pred_loss_m

        logit_scale_num = 0.5
        
        #if self.use_sample_contras_loss:
        if use_sample_loss:
            a = 0.6
            sample_loss = self.contras_loss(feature_s, feature_m, gt, logit_scale_num)
            loss = a * sample_loss + loss
        if self.use_class_contras_loss:
            b = 0.4
            class_loss = self.class_contras_loss(feature_s, feature_m, feature_class, gt, logit_scale_num)
            loss = b * class_loss + loss
         
        pred = ret_s.argmax(-1)
        acc = (pred == gt).sum() / float(gt.size(0))
        return loss, acc * 100

    def load_model_from_ckpt(self, bert_ckpt_path):
        if bert_ckpt_path is not None:
            ckpt = torch.load(bert_ckpt_path)
            base_ckpt = {k.replace("module.", ""): v for k, v in ckpt['base_model'].items()}

            for k in list(base_ckpt.keys()):
                if k.startswith('MAE_encoder') :
                    base_ckpt[k[len('MAE_encoder.'):]] = base_ckpt[k]
                    del base_ckpt[k]
                elif k.startswith('base_model'):
                    base_ckpt[k[len('base_model.'):]] = base_ckpt[k]
                    del base_ckpt[k]

            incompatible = self.load_state_dict(base_ckpt, strict=False)

            if incompatible.missing_keys:
                print_log('missing_keys', logger='Transformer')
                print_log(
                    get_missing_parameters_message(incompatible.missing_keys),
                    logger='Transformer'
                )
            if incompatible.unexpected_keys:
                print_log('unexpected_keys', logger='Transformer')
                print_log(
                    get_unexpected_parameters_message(incompatible.unexpected_keys),
                    logger='Transformer'
                )

            print_log(f'[Transformer] Successful Loading the ckpt from {bert_ckpt_path}', logger='Transformer')
        else:
            print_log('Training from scratch!!!', logger='Transformer')
            self.apply(self._init_weights)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)
        elif isinstance(m, nn.Conv1d):
            trunc_normal_(m.weight, std=.02)
            if m.bias is not None:
                nn.init.constant_(m.bias, 0)

    def forward(self, pts, mode='standard', mask=None, vis=False):

        neighborhood, center = self.group_divider(pts)
        group_input_tokens = self.encoder(neighborhood)  # B G N

        cls_tokens = self.cls_token.expand(group_input_tokens.size(0), -1, -1)
        cls_pos = self.cls_pos.expand(group_input_tokens.size(0), -1, -1)

        if mode == 'standard':
            pos = self.pos_embed(center)
            
            pos = torch.cat((cls_pos, pos), dim=1)
            x = torch.cat((cls_tokens, group_input_tokens), dim=1)

            # transformer
            x, mask, score = self.blocks(x, pos, self.mask_ratio)
            x = self.norm(x)
            concat_f = torch.cat([x[:, 0], x[:, 1:].max(1)[0]], dim=-1)
            ret = self.cls_head_finetune(concat_f)
            if vis: #visualization
                """
                vis_points = neighborhood[~mask].reshape(B * (self.num_group - M), -1, 3)
                full_vis = vis_points + center[~mask].unsqueeze(1)
                full_rebuild = rebuild_points + center[mask].unsqueeze(1)
                full = torch.cat([full_vis, full_rebuild], dim=0)
                # full_points = torch.cat([rebuild_points,vis_points], dim=0)
                full_center = torch.cat([center[mask], center[~mask]], dim=0)
                # full = full_points + full_center.unsqueeze(1)
                ret2 = full_vis.reshape(-1, 3).unsqueeze(0)
                ret1 = full.reshape(-1, 3).unsqueeze(0)
                """
                pc_points = neighborhood + center.unsqueeze(2)

                return pc_points, center, score
            
            return concat_f, ret, mask
        else:
            B, G, C = group_input_tokens.size()

            vis_tokens = group_input_tokens[~mask].reshape(B, -1, C)
            _,N,_ = vis_tokens.shape
            M = G-N
            #mask_tokens = self.attn_mask_token.expand(B, M, -1)
            mask_tokens = self.mask_token.expand(B, M, -1)
            #mask_pos = self.mask_pos.expand(B, M, -1)
            full_tokens = torch.cat([vis_tokens, mask_tokens], dim=1)
                       
            vis_pos = self.pos_embed(center[~mask]).reshape(B, -1, C)
            mask_pos = self.pos_embed(center[mask]).reshape(B, -1, C)
            full_pos = torch.cat((vis_pos, mask_pos), dim=1)
            
            x = torch.cat((cls_tokens, full_tokens), dim=1)
            pos = torch.cat((cls_pos, full_pos), dim=1)          
            
            # transformer
            x, mask, _ = self.blocks(x, pos, self.mask_ratio)
            x = self.norm(x)
            concat_f = torch.cat([x[:, 0], x[:, 1:].max(1)[0]], dim=-1)
            ret = self.cls_head_finetune(concat_f)
            return concat_f, ret, mask

