#!/usr/bin/env python3
import argparse
import os
import json
import torch
import dill
from counting_sampling_v2 import AutomatonBuilder


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(description="Arguments for automaton generation")
    parser.add_argument("--num_states", type=int, default=20)
    parser.add_argument("--num_symbols", type=int, default=10)
    parser.add_argument("--accept_prob", type=float, default=0.2)
    parser.add_argument("--topology_seed", type=int, default=42, 
                        help="Seed for automaton topology generation")
    parser.add_argument("--seed", type=int, default=42,
                        help="General seed (for backwards compatibility)")
    parser.add_argument("--automaton_name", type=str,
                        help="Lookup a specific automaton from an external register")
    parser.add_argument("--output_dir", type=str, default="data")
    parser.add_argument("--save_automaton", action="store_true", default=True)
    return parser.parse_args()


def main(args=None):
    if args is None:
        args = parse_args()
    print(args)

    # Use topology_seed if provided, otherwise fall back to the general seed
    seed_to_use = args.topology_seed if hasattr(args, 'topology_seed') else args.seed
    
    # Create a dictionary of common sampling parameters to persist
    sample_args = {
        "seed": seed_to_use,
        "num_states": args.num_states,
        "accept_prob": args.accept_prob,
        "automaton_name": args.automaton_name,
        "alphabet_size": args.num_symbols
    }
    
    # Store the seed used for reference in later steps
    seed_info = {
        "topology_seed": seed_to_use
    }
    
    # Step 1: Build the automaton using AutomatonBuilder from our modular implementation
    builder = AutomatonBuilder(
        args.num_states, 
        args.num_symbols,
        seed_to_use,
        args.accept_prob,
        args.automaton_name
    )
    automaton = builder.build()

    # Create output directory
    os.makedirs(args.output_dir, exist_ok=True)
    
    # Save the built automaton
    if args.save_automaton:
        automaton_path = os.path.join(args.output_dir, "automaton.pkl")
        with open(automaton_path, "wb") as f:
            dill.dump(automaton, f)
    
    # Save sampling parameters for next step
    sample_args_path = os.path.join(args.output_dir, "sample_args.json")
    with open(sample_args_path, "w") as f:
        json.dump({k: v for k, v in sample_args.items() if isinstance(v, (int, float, str, bool, list, dict))}, f)
    
    # Save seed information for later steps
    seed_info_path = os.path.join(args.output_dir, "seed_info.json")
    with open(seed_info_path, "w") as f:
        json.dump(seed_info, f)
    
    print(f"Automaton topology generated and saved to {args.output_dir}")


if __name__ == "__main__":
    main()