import math
import torch
import torch.nn as nn

from transformers.models.mistral.modeling_mistral import (
    MistralModel,
    MistralAttention,
    MistralForCausalLM,
    MistralConfig,
    BaseModelOutputWithPast,
    DynamicCache,
    Cache,
    FlashAttentionKwargs,
    Unpack,
    ALL_ATTENTION_FUNCTIONS,
    eager_attention_forward,
    apply_rotary_pos_emb,
    logger,
    repeat_kv
)
from typing import (
    Callable,
    Optional,
    Tuple,
    Union,
    List,
    Dict,
    Any
)

from src.utils import rotate_v_proj
from src.utils import rotate_o_proj
from src.utils import apply_rotary_pos_emb_single
from src.models.cache import NSNCache
from src.quantizers.quantizer_factory import get_quantizer
from src.quantizers.base_quantizer import BaseQuantizer


def get_mistral_model(model_name_or_path, quant_config, forward_quant, device):
    """
    Get a quantized Mistral model.

    Args:
        model_name_or_path (str): The name or path of the model to load.
        quant_config (dict): The quantization configuration.
        forward_quant (bool): Whether to use forward quantization.
        device (str): The device to run the model on.

    Caution: ONLY supports Mistral-7B-Instruct-v0.2 and Mistral-7B-Instruct-v0.3. v0.1 needs sliding window cache.
    """
    device = device or "auto"
    model = QuantizedMistralForCausalLM.from_pretrained(model_name_or_path,
                                                        device_map=device,
                                                        forward_quant=forward_quant,
                                                        quant_config=quant_config,
                                                        torch_dtype=torch.half)
    model = model.half().eval()
    if (quant_config["kwargs"] is not None) and("hadamard" in quant_config["kwargs"]) and quant_config["kwargs"]["hadamard"]:
        for layer in model.model.layers:
            rotate_v_proj(layer.self_attn.v_proj, layer.self_attn.head_dim)
            rotate_o_proj(layer.self_attn.o_proj, layer.self_attn.head_dim)
    return model


class QuantizedMistralModel(MistralModel):
    cache_class = DynamicCache
    def forward(
        self,
        input_ids: torch.LongTensor = None,
        attention_mask: Optional[torch.Tensor] = None,
        position_ids: Optional[torch.LongTensor] = None,
        past_key_values: Optional[Cache] = None,
        inputs_embeds: Optional[torch.FloatTensor] = None,
        use_cache: Optional[bool] = None,
        output_attentions: Optional[bool] = None,
        output_hidden_states: Optional[bool] = None,
        return_dict: Optional[bool] = None,
        cache_position: Optional[torch.LongTensor] = None,
        **flash_attn_kwargs: Unpack[FlashAttentionKwargs],
    ) -> Union[Tuple, BaseModelOutputWithPast]:
        output_attentions = output_attentions if output_attentions is not None else self.config.output_attentions
        output_hidden_states = (
            output_hidden_states if output_hidden_states is not None else self.config.output_hidden_states
        )
        use_cache = use_cache if use_cache is not None else self.config.use_cache
        if use_cache and ((past_key_values is None) or (len(past_key_values)==0)):
            past_key_values = self.cache_class()
        return_dict = return_dict if return_dict is not None else self.config.use_return_dict

        if (input_ids is None) ^ (inputs_embeds is not None):
            raise ValueError("You must specify exactly one of input_ids or inputs_embeds")

        if self.gradient_checkpointing and self.training and use_cache:
            logger.warning_once(
                "`use_cache=True` is incompatible with gradient checkpointing. Setting `use_cache=False`."
            )
            use_cache = False

        if inputs_embeds is None:
            inputs_embeds = self.embed_tokens(input_ids)

        if use_cache and past_key_values is None:
            past_key_values = DynamicCache()

        if cache_position is None:
            past_seen_tokens = past_key_values.get_seq_length() if past_key_values is not None else 0
            cache_position = torch.arange(
                past_seen_tokens, past_seen_tokens + inputs_embeds.shape[1], device=inputs_embeds.device
            )

        if position_ids is None:
            position_ids = cache_position.unsqueeze(0)

        causal_mask = self._update_causal_mask(
            attention_mask, inputs_embeds, cache_position, past_key_values, output_attentions
        )

        hidden_states = inputs_embeds

        # create position embeddings to be shared across the decoder layers
        # print(position_ids)
        offset = position_ids.amax() - position_ids.amax(dim=1)
        cos, sin = self.rotary_emb(hidden_states, position_ids)
        position_embeddings = (cos, sin, offset.int(), self.rotary_emb.inv_freq)

        # decoder layers
        all_hidden_states = () if output_hidden_states else None
        all_self_attns = () if output_attentions else None

        for decoder_layer in self.layers[: self.config.num_hidden_layers]:
            if output_hidden_states:
                all_hidden_states += (hidden_states,)

            if self.gradient_checkpointing and self.training:
                layer_outputs = self._gradient_checkpointing_func(
                    decoder_layer.__call__,
                    hidden_states,
                    causal_mask,
                    position_ids,
                    past_key_values,
                    output_attentions,
                    use_cache,
                    cache_position,
                    position_embeddings,
                )
            else:
                layer_outputs = decoder_layer(
                    hidden_states,
                    attention_mask=causal_mask,
                    position_ids=position_ids,
                    past_key_value=past_key_values,
                    output_attentions=output_attentions,
                    use_cache=use_cache,
                    cache_position=cache_position,
                    position_embeddings=position_embeddings,
                    **flash_attn_kwargs,
                )

            hidden_states = layer_outputs[0]

            if output_attentions:
                all_self_attns += (layer_outputs[1],)

        hidden_states = self.norm(hidden_states)

        # add hidden states from the last decoder layer
        if output_hidden_states:
            all_hidden_states += (hidden_states,)

        output = BaseModelOutputWithPast(
            last_hidden_state=hidden_states,
            past_key_values=past_key_values if use_cache else None,
            hidden_states=all_hidden_states,
            attentions=all_self_attns,
        )
        return output if return_dict else output.to_tuple()

class QuantizedMistralForCausalLM(MistralForCausalLM):
    forward_quant: bool = False
    quant_config: Dict = None
    def __init__(self, config: MistralConfig):
        super().__init__(config)
        self.model = QuantizedMistralModel(config)
        self.vocab_size = config.vocab_size
        self.lm_head = nn.Linear(config.hidden_size, config.vocab_size, bias=False)

        # Initialize weights and apply final processing
        self.post_init()
        self._replace_attention()
        self._set_cache_class()

    @classmethod
    def from_pretrained(cls, *args, **kwargs):
        if "quant_config" in kwargs:
            cls.quant_config = kwargs["quant_config"]
            kwargs.pop("quant_config")
        if "forward_quant" in kwargs:
            cls.forward_quant = kwargs["forward_quant"]
            kwargs.pop("forward_quant")
        return super().from_pretrained(*args, **kwargs)

    def _replace_attention(self):
        for layer in self.model.layers:
            prev_attn = layer.self_attn
            new_attn = QuantizedMistralAttention(
                prev_attn.config,
                prev_attn.layer_idx,
                self.forward_quant,
                get_quantizer(self.quant_config, prev_attn.layer_idx)
            )
            layer.self_attn = new_attn
            del prev_attn

    def _set_cache_class(self):
        cache_dict = {
            "NSNQuantizer": NSNCache,
            "KIVIQuantizer": DynamicCache,
            "Identity": DynamicCache,
            "KVQuantQuantizer": NSNCache,
            "CoupledQuantizer": NSNCache,
        }
        self.model.cache_class = cache_dict[self.quant_config["name"]]
        

class QuantizedMistralAttention(MistralAttention):
    def __init__(self, config: MistralConfig, layer_idx: int, forward_quant: bool, quantizer: BaseQuantizer):
        super().__init__(config, layer_idx)
        self.forward_quant = forward_quant
        self.quantizer = quantizer

    def forward(
        self,
        hidden_states: torch.Tensor,
        position_embeddings: Tuple[torch.Tensor, torch.Tensor],
        attention_mask: Optional[torch.Tensor],
        past_key_value: Optional[Cache] = None,
        cache_position: Optional[torch.LongTensor] = None,
        **kwargs: Unpack[FlashAttentionKwargs],
    ) -> Tuple[torch.Tensor, Optional[torch.Tensor], Optional[Tuple[torch.Tensor]]]:
        attention_interface: Callable = eager_attention_forward
        if self.config._attn_implementation != "eager":
            if self.config._attn_implementation == "sdpa" and kwargs.get("output_attentions", False):
                logger.warning_once(
                    "`torch.nn.functional.scaled_dot_product_attention` does not support `output_attentions=True`. Falling back to "
                    'eager attention. This warning can be removed using the argument `attn_implementation="eager"` when loading the model.'
                )
            else:
                attention_interface = ALL_ATTENTION_FUNCTIONS[self.config._attn_implementation]

        input_shape = hidden_states.shape[:-1]
        hidden_shape = (*input_shape, -1, self.head_dim)

        query_states = self.q_proj(hidden_states).view(hidden_shape).transpose(1, 2)
        key_states = self.k_proj(hidden_states).view(hidden_shape).transpose(1, 2)
        value_states = self.v_proj(hidden_states).view(hidden_shape).transpose(1, 2).contiguous()

        cos, sin, offset, inv_freq = position_embeddings
        if self.forward_quant:
            query_states, key_states, value_states, is_prefill = self.quantizer.forward_quant(query_states,
                                                                                              key_states,
                                                                                              value_states,
                                                                                              position_embeddings,
                                                                                              past_key_value,
                                                                                              cache_position)
        else:
            if past_key_value is not None:
                cache_kwargs = {"sin": sin, "cos": cos, "cache_position": cache_position}
                is_prefill = False if (len(past_key_value) > self.layer_idx) else True
                if is_prefill:
                    for i in range(offset.shape[0]):
                        for j in range(offset[i]):
                            key_states[i, :, j] = key_states[i, :, offset[i]]
                            value_states[i, :, j] = value_states[i, :, offset[i]]

                if self.quantizer.post_rope:
                    query_states, key_states = apply_rotary_pos_emb(query_states, key_states, cos, sin)
                    key_states, value_states = past_key_value.update(key_states, value_states, self.layer_idx, cache_kwargs)
                else:
                    # Applying RoPE is performed in self_attn method of quantizer
                    query_states = apply_rotary_pos_emb_single(query_states, cos, sin)
                    key_states, value_states = past_key_value.update(key_states, value_states, self.layer_idx, cache_kwargs)
                    if is_prefill:
                        key_states = apply_rotary_pos_emb_single(key_states, cos, sin)
            else:
                is_prefill = True
        

        if is_prefill:
            attn_output, attn_weights = attention_interface(
                self,
                query_states,
                key_states,
                value_states,
                attention_mask,
                dropout=0.0 if not self.training else self.attention_dropout,
                scaling=self.scaling,
                **kwargs,
            )
        else:
            attn_output, attn_weights = self.quantizer.self_attn(query_states,
                                                                 past_key_value,
                                                                 self.layer_idx,
                                                                 attention_mask,
                                                                 self.scaling,
                                                                 self.num_key_value_groups,
                                                                 inv_freq=inv_freq,
                                                                 offset=offset)

        if past_key_value is not None and self.quantizer and not self.forward_quant: # Execute only when evaluating ppl
            self.quantizer.update_cache(past_key_value, self.layer_idx, is_prefill)

        attn_output = attn_output.reshape(*input_shape, -1).contiguous()
        attn_output = self.o_proj(attn_output)
        return attn_output, attn_weights
