import os
import contextlib
from detectron2.data import DatasetCatalog, MetadataCatalog
from fvcore.common.timer import Timer
from fvcore.common.file_io import PathManager
import io
import logging
from detectron2.data.datasets.pascal_voc import register_pascal_voc
from .my_pascal_voc import my_register_pascal_voc

logger = logging.getLogger(__name__)

JSON_ANNOTATIONS_DIR = ""
_SPLITS_COCO_FORMAT = {}
_SPLITS_COCO_FORMAT["coco"] = {
    "coco_2017_unlabel": (
        "coco/unlabeled2017",
        "coco/annotations/image_info_unlabeled2017.json",
    ),
    "coco_2017_for_voc20": (
        "coco",
        "coco/annotations/google/instances_unlabeledtrainval20class.json",
    ),
}


def register_coco_unlabel(root):
    for _, splits_per_dataset in _SPLITS_COCO_FORMAT.items():
        for key, (image_root, json_file) in splits_per_dataset.items():
            meta = {}
            register_coco_unlabel_instances(
                key, meta, os.path.join(root, json_file), os.path.join(root, image_root)
            )


def register_coco_unlabel_instances(name, metadata, json_file, image_root):
    """
    Register a dataset in COCO's json annotation format for
    instance detection, instance segmentation and keypoint detection.
    (i.e., Type 1 and 2 in http://cocodataset.org/#format-data.
    `instances*.json` and `person_keypoints*.json` in the dataset).

    This is an example of how to register a new dataset.
    You can do something similar to this function, to register new datasets.

    Args:
        name (str): the name that identifies a dataset, e.g. "coco_2014_train".
        metadata (dict): extra metadata associated with this dataset.  You can
            leave it as an empty dict.
        json_file (str): path to the json instance annotation file.
        image_root (str or path-like): directory which contains all the images.
    """
    assert isinstance(name, str), name
    assert isinstance(json_file, (str, os.PathLike)), json_file
    assert isinstance(image_root, (str, os.PathLike)), image_root

    # 1. register a function which returns dicts
    DatasetCatalog.register(
        name, lambda: load_coco_unlabel_json(json_file, image_root, name)
    )

    # 2. Optionally, add metadata about this dataset,
    # since they might be useful in evaluation, visualization or logging
    MetadataCatalog.get(name).set(
        json_file=json_file, image_root=image_root, evaluator_type="coco", **metadata
    )


def load_coco_unlabel_json(
    json_file, image_root, dataset_name=None, extra_annotation_keys=None
):
    from pycocotools.coco import COCO

    timer = Timer()
    json_file = PathManager.get_local_path(json_file)
    with contextlib.redirect_stdout(io.StringIO()):
        coco_api = COCO(json_file)
    if timer.seconds() > 1:
        logger.info(
            "Loading {} takes {:.2f} seconds.".format(json_file, timer.seconds())
        )

    id_map = None
    # sort indices for reproducible results
    img_ids = sorted(coco_api.imgs.keys())

    imgs = coco_api.loadImgs(img_ids)

    logger.info("Loaded {} images in COCO format from {}".format(len(imgs), json_file))

    dataset_dicts = []

    for img_dict in imgs:
        record = {}
        record["file_name"] = os.path.join(image_root, img_dict["file_name"])
        record["height"] = img_dict["height"]
        record["width"] = img_dict["width"]
        image_id = record["image_id"] = img_dict["id"]

        dataset_dicts.append(record)

    return dataset_dicts


# ==== Predefined splits for PASCAL VOC ===========
def register_all_pascal_voc(root):
    SPLITS = [
        ("VOC2007_citytrain", 'data/VOC2007_citytrain', "train", 8),
        ("VOC2007_foggytrain", 'data/VOC2007_foggytrain', "train", 8),
        ("VOC2007_foggyval", 'data/VOC2007_foggyval', "val", 8),
        ("VOC2007_citytrain1", 'data/VOC2007_citytrain1', "train", 1),
        ("VOC2007_cityval1", 'data/VOC2007_cityval1', "val", 1),
        ("VOC2007_bddtrain", 'data/VOC2007_bddtrain', "train", 8),
        ("VOC2007_bddval", 'data/VOC2007_bddval', "val", 8),
        ("VOC2007_kitti1", 'data/kitti', "train", 1),
        ("VOC2007_sim1", 'data/sim', "train", 1),
        ("SDGOD_daytime_clear_train", 'data/daytime_clear', "train", 7),
        ("SDGOD_daytime_foggy_trainval", 'data/daytime_foggy', "trainval", 7),
        ("SDGOD_daytime_foggy_train", 'data/daytime_foggy', "train", 7),
        ("SDGOD_daytime_foggy_test", 'data/daytime_foggy', "test", 7),
        ("SDGOD_dusk_rainy_train", 'data/dusk_rainy', "train", 7),
        ("SDGOD_night_rainy_train", 'data/night_rainy', "train", 7),
        ("SDGOD_night_sunny_trainval", 'data/night_sunny', "trainval", 7),
        ("SDGOD_night_sunny_train", 'data/night_sunny', "train", 7),
        ("SDGOD_night_sunny_test", 'data/night_sunny', "test", 7),
        ("crossdet_clipart_train", 'data/clipart', "train", 6),
        ("crossdet_clipart_test", 'data/clipart', "test", 6),
        ("crossdet_comic_train", 'data/comic', "train", 6),
        ("crossdet_comic_test", 'data/comic', "test", 6),
        ("crossdet_watercolor_train", 'data/watercolor', "train", 6),
        ("crossdet_watercolor_test", 'data/watercolor', "test", 6),
        ("crossdet_voc07_train", 'data/voc07', "train", 6),
        ("crossdet_voc07_trainval", 'data/voc07', "trainval", 6),
        ("crossdet_voc07_test", 'data/voc07', "test", 6),
        ("crossdet_voc12_train", 'data/voc12', "train", 6),
        ("crossdet_voc12_trainval", 'data/voc12', "trainval", 6),
        ("crossdet_voc12_test", 'data/voc12', "test", 6),
    ]
    for name, dirname, split, cls in SPLITS:
        year = 2012
        if 'VOC2007' in name:
            if cls == 1:
                class_names = ('car',)
            elif cls == 20:
                class_names = (
                    "aeroplane", "bicycle", "bird", "boat", "bottle", "bus", "car", "cat",
                    "chair", "cow", "diningtable", "dog", "horse", "motorbike", "person",
                    "pottedplant", "sheep", "sofa", "train", "tvmonitor"
                )
            elif cls == 8:
                class_names = ('truck', 'car', 'rider', 'person', 'train', 'motorcycle', 'bicycle', 'bus')
            elif cls == 7:
                class_names = ('truck', 'car', 'rider', 'person', 'motorcycle', 'bicycle', 'bus')
            else:
                raise RuntimeError
        elif 'SDGOD' in name:
            class_names = ('bus', 'bike', 'car', 'motor', 'person', 'rider', 'truck')
        elif 'crossdet' in name:
            class_names = ('bicycle', 'bird', 'cat', 'car', 'dog', 'person')
        else:
            raise RuntimeError('Wrong dataset name.')
        
        if 'crossdet' in name:
            my_register_pascal_voc(name, os.path.join(root, dirname), split, year, class_names)
        else:
            register_pascal_voc(name, os.path.join(root, dirname), split, year, class_names)
        MetadataCatalog.get(name).evaluator_type = "pascal_voc"


_root = os.getenv("DETECTRON2_DATASETS", "")
register_coco_unlabel(_root)
register_all_pascal_voc(_root)
