from math import factorial
from operator import itemgetter
from itertools import combinations

import numpy as np

def get_coalition_values(model, Xs, noises=None):
    n = len(Xs)
    coalitions = [list(j) for i in range(n) for j in combinations(list(range(n)), i+1)] # no empty set
    v = {frozenset(subset): cig(subset, model, Xs, noises) for subset in coalitions}
    return v

def cig(coalition, model, Xs, noises=None):
    n = len(Xs)
    complement = [i for i in range(n) if i not in coalition]
    if complement:
        X = np.vstack(itemgetter(*complement)(Xs))
        noise = np.vstack(itemgetter(*complement)(noises)) if noises else None
        ig_complement = model.mi(X, noise)
    else:
        ig_complement = 0
    ig_grand_coalition = model.mi(np.vstack(Xs), np.vstack(noises) if noises else None)
    return ig_grand_coalition - ig_complement

def get_shapley_value(i, v):
    phi = 0.0
    n = max(len(s) for s in v.keys())
    for (coalition, value) in v.items():
        if i not in coalition:
            weight = factorial(len(coalition)) * factorial(n - len(coalition) - 1) / factorial(n)
            marginal_contribution = v[coalition.union({i})] - value
            phi += weight * marginal_contribution
    weight = factorial(0) * factorial(n-1) / factorial(n)
    phi += weight * v[frozenset([i])]
    return phi

def get_shapley_values(v):
    n = max(len(s) for s in v.keys())
    shapleys = [0.0] * n
    for i in range(n):
        shapleys[i] = get_shapley_value(i, v)
    return shapleys

def get_dividends(v):
    dividends = {}
    for coalition in v.keys():
        dividend = v[coalition]
        for subset in dividends.keys():
            if subset < coalition:
                dividend -= dividends[subset]
        dividends[coalition] = dividend
    return dividends

def get_time_aware_v(v, times, weight_function):
    weights = weight_function(times)
    dividends = get_dividends(v)
    time_aware_v = {}
    for coalition in v.keys():
        value = 0.0
        for subset in dividends.keys():
            if subset <= coalition:
                if len(subset) == 1:
                    value += dividends[subset]
                else:
                    min_weight = min([w for (i, w) in enumerate(weights) if i in subset])
                    value += min_weight * dividends[subset]
        time_aware_v[coalition] = value
    return time_aware_v

def get_N_tau(times, tau):
    n = len(times)
    N_tau = [i for i in range(n) if times[i] <= tau]
    return frozenset(set(N_tau))

def get_N_tau_complement(times, tau):
    n = len(times)
    N_tau = [i for i in range(n) if times[i] > tau]
    return frozenset(set(N_tau))

def get_v_tau(v, N_tau):
    v_tau = {}
    for coalition in v.keys():
        if coalition <= N_tau:
            v_tau[coalition] = v[coalition]
    return v_tau

def get_reward_cumulation(v, times, beta):
    n = max(len(s) for s in v.keys())
    rewards = [0.0] * n
    w_sum = 0.0
    T = int(max(times))
    for tau in range(T+1):
        w_tau = beta**tau
        w_sum += w_tau
        N_tau = get_N_tau(times, tau)
        N_tau_complement = get_N_tau_complement(times, tau)
        v_tau = get_v_tau(v, N_tau)
        for i in N_tau:
            rewards[i] += w_tau * get_shapley_value(i, v_tau)
        for i in N_tau_complement:
            rewards[i] += w_tau * v[frozenset([i])]
    rewards = [reward / w_sum for reward in rewards]
    return rewards

def get_scaling_factor(v):
    v_N = max(v.values())
    v_max = max(get_shapley_values(v))
    return v_N / v_max
