import argparse

from data import get_data_and_model
from valuation import get_coalition_values
from reward import get_reward_value, get_reward_subset, get_reward_noise
from plot import plot_value, plot_subset, plot_noise

def main(args):
    gp, X_splits, y_splits, noise_splits, X_test, y_test, noise_test = get_data_and_model(args.dataset, args.scenario)
    v = get_coalition_values(gp, X_splits, noise_splits)
    gamma_list = [0, 0.5, 1]
    beta_list = [0.1, 0.7, 1]
    plotdir = f'{args.dataset}_{args.scenario}'
    if args.method == 'value':
        df_value = get_reward_value(args.dataset, args.scenario, v=v, gamma_list=gamma_list, beta_list=beta_list)
        plot_value(df_value, v, gamma_list, beta_list, plotdir, format=args.format)
        print("Finished plotting reward values without realization.")
    elif args.method == 'noise':
        df_noise = get_reward_noise(args.dataset, args.scenario, gamma_list=gamma_list, beta_list=beta_list, gp=gp, X_splits=X_splits, y_splits=y_splits, noise_splits=noise_splits, X_test=X_test, y_test=y_test, noise_test=noise_test, v=v)
        plot_noise(df_noise, gamma_list, beta_list, plotdir, format=args.format)
        print("Finished plotting reward realization via likelihood tempering.")
    elif args.method == 'subset':
        df_subset = get_reward_subset(args.dataset, args.scenario, gamma_list=gamma_list, beta_list=beta_list, gp=gp, X_splits=X_splits, y_splits=y_splits, noise_splits=noise_splits, X_test=X_test, y_test=y_test, noise_test=noise_test, v=v)
        plot_subset(df_subset, gamma_list, beta_list, plotdir, format=args.format)
        print("Finished plotting reward realization via subset selection.")

def parse_arguments():
    parser = argparse.ArgumentParser(description="run experiments and plot results",
                                     formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument('--dataset', type=str, default='friedman', choices=['friedman', 'diap', 'calih'])
    parser.add_argument('--scenario', type=int, default=0, choices=[0, 1, 2], 
                        help="scenarios of dataset partitions, check config.yaml for details")
    parser.add_argument('--method', type=str, default='value', choices=['value', 'noise', 'subset'], 
                        help=("reward realization methods\n"
                              "  -value: (default) reward values, no realization\n"
                              "  -noise: likelihood tempering\n"
                              "  -subset: subset selection"))
    parser.add_argument('--format', type=str, default='png', choices=['png', 'pdf'],
                        help="format of saved plots")
    args = parser.parse_args()
    return args

if __name__ == '__main__':
    args = parse_arguments()
    main(args)
