import logging

import torch
import torch.nn as nn
import torch.nn.functional as F
from collections import OrderedDict

logger = logging.getLogger(__name__)


def mish(x):
    """Mish: A Self Regularized Non-Monotonic Neural Activation Function (https://arxiv.org/abs/1908.08681)"""
    return x * torch.tanh(F.softplus(x))


class PSBatchNorm2d(nn.BatchNorm2d):
    """How Does BN Increase Collapsed Neural Network Filters? (https://arxiv.org/abs/2001.11216)"""

    def __init__(self, num_features, alpha=0.1, eps=1e-05, momentum=0.001, affine=True, track_running_stats=True):
        super().__init__(num_features, eps, momentum, affine, track_running_stats)
        self.alpha = alpha

    def forward(self, x):
        return super().forward(x) + self.alpha


class BasicBlock(nn.Module):
    def __init__(self, in_planes, out_planes, stride, drop_rate=0.0, activate_before_residual=False):
        super(BasicBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes, momentum=0.001)
        self.relu1 = nn.LeakyReLU(negative_slope=0.1, inplace=True)
        self.conv1 = nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                               padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(out_planes, momentum=0.001)
        self.relu2 = nn.LeakyReLU(negative_slope=0.1, inplace=True)
        self.conv2 = nn.Conv2d(out_planes, out_planes, kernel_size=3, stride=1,
                               padding=1, bias=False)
        self.drop_rate = drop_rate
        self.equalInOut = (in_planes == out_planes)
        self.convShortcut = (not self.equalInOut) and nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride,
                                                                padding=0, bias=False) or None
        self.activate_before_residual = activate_before_residual

    def forward(self, x):
        if not self.equalInOut and self.activate_before_residual == True:
            x = self.relu1(self.bn1(x))
        else:
            out = self.relu1(self.bn1(x))
        out = self.relu2(self.bn2(self.conv1(out if self.equalInOut else x)))
        if self.drop_rate > 0:
            out = F.dropout(out, p=self.drop_rate, training=self.training)
        out = self.conv2(out)
        return torch.add(x if self.equalInOut else self.convShortcut(x), out)


class NetworkBlock(nn.Module):
    def __init__(self, nb_layers, in_planes, out_planes, block, stride, drop_rate=0.0, activate_before_residual=False):
        super(NetworkBlock, self).__init__()
        self.layer = self._make_layer(
            block, in_planes, out_planes, nb_layers, stride, drop_rate, activate_before_residual)

    def _make_layer(self, block, in_planes, out_planes, nb_layers, stride, drop_rate, activate_before_residual):
        layers = []
        for i in range(int(nb_layers)):
            layers.append(block(i == 0 and in_planes or out_planes, out_planes,
                                i == 0 and stride or 1, drop_rate, activate_before_residual))
        return nn.Sequential(*layers)

    def forward(self, x):
        return self.layer(x)


class WideResNet(nn.Module):
    def __init__(self, num_classes, depth=28, widen_factor=2, drop_rate=0.0,
                head='mlp', head_depth=2, hidden_dim=512, feat_dim=512):
        super(WideResNet, self).__init__()
        channels = [16, 16*widen_factor, 32*widen_factor, 64*widen_factor]
        assert((depth - 4) % 6 == 0)
        n = (depth - 4) / 6
        block = BasicBlock
        # 1st conv before any network block
        self.conv1 = nn.Conv2d(3, channels[0], kernel_size=3, stride=1,
                               padding=1, bias=False)
        # 1st block
        self.block1 = NetworkBlock(
            n, channels[0], channels[1], block, 1, drop_rate, activate_before_residual=True)
        # 2nd block
        self.block2 = NetworkBlock(
            n, channels[1], channels[2], block, 2, drop_rate)
        # 3rd block
        self.block3 = NetworkBlock(
            n, channels[2], channels[3], block, 2, drop_rate)
        # global average pooling and classifier
        self.bn1 = nn.BatchNorm2d(channels[3], momentum=0.001)
        self.relu = nn.LeakyReLU(negative_slope=0.1, inplace=True)
        # if not non_linear:
        #     self.fc = nn.Linear(channels[3], num_classes)
        # else:
        #     list_layers = [nn.Linear(channels[3],channels[3]), nn.LeakyReLU(negative_slope=0.1, inplace=True)]
        #     for _ in range(depth-2):
        #         list_layers += [nn.Linear(channels[3],channels[3]), nn.LeakyReLU(negative_slope=0.1, inplace=True)]
        #     list_layers += [nn.Linear(channels[3],num_classes)]
        #     print("using MLP as Classifier with hidden dim={} and depth={}".format(channels[3],depth))
        #     self.fc = nn.Sequential(*list_layers)

        self.channels = channels[3]



        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight,
                                        mode='fan_out',
                                        nonlinearity='leaky_relu')
            elif isinstance(m, nn.BatchNorm2d):
                nn.init.constant_(m.weight, 1.0)
                nn.init.constant_(m.bias, 0.0)
            elif isinstance(m, nn.Linear):
                nn.init.xavier_normal_(m.weight)
                nn.init.constant_(m.bias, 0.0)

    def forward(self, x):
        out = self.conv1(x)
        out = self.block1(out)
        out = self.block2(out)
        out = self.block3(out)
        out = self.relu(self.bn1(out))
        out = F.adaptive_avg_pool2d(out, 1)
        out = out.view(-1, self.channels)
        return out


def build_wideresnet(depth, widen_factor, dropout, num_classes, version='fixmatch'):
    logger.info(f"Model: WideResNet {depth}x{widen_factor}; version {version}")
    if version == 'fixmatch':
        return WideResNet(depth=depth,
                      widen_factor=widen_factor,
                      drop_rate=dropout,
                      num_classes=num_classes)
    elif version == 'paws':
        return PawsWideResNet(depth=depth,
                      widen_factor=widen_factor,
                      drop_rate=dropout)

class WideResNetProj(nn.Module):
    """encoder + classifier"""
    def __init__(self, num_classes=10, hidden_dim=512,output_dim=512, depth=28, widen_factor=2, dropout=0.0, non_linear=False, clas_depth=2, version='fixmatch'):
        super(WideResNetProj, self).__init__()
        self.encoder = build_wideresnet(depth=depth,
                          widen_factor=widen_factor,
                          dropout=dropout,
                          num_classes=num_classes,
                          version=version)
        self.dim_in = self.encoder.channels

        self.fc = torch.nn.Sequential(OrderedDict([
            ('fc1', torch.nn.Linear(self.dim_in, hidden_dim)),
            ('bn1', torch.nn.BatchNorm1d(hidden_dim)),
            ('relu1', torch.nn.ReLU(inplace=True)),
            ('fc2', torch.nn.Linear(hidden_dim, hidden_dim)),
            ('bn2', torch.nn.BatchNorm1d(hidden_dim)),
            ('relu2', torch.nn.ReLU(inplace=True)),
            ('fc3', torch.nn.Linear(hidden_dim, output_dim))
        ]))

    def forward(self, inputs):
        if not isinstance(inputs, list):
            inputs = [inputs]
        idx_crops = torch.cumsum(torch.unique_consecutive(
            torch.tensor([inp.shape[-1] for inp in inputs]),
            return_counts=True,
        )[1], 0)

        start_idx = 0
        for end_idx in idx_crops:
            _z = self.encoder(torch.cat(inputs[start_idx:end_idx]))

            if start_idx == 0:
                z = _z
            else:
                z = torch.cat((z, _z))
            start_idx = end_idx

        return z

class Online_Classifier(nn.Module):
    def __init__(self, num_classes=100, model_name='wide_resnet28w8', mlp=False, depth=2, dim_in=-1):
        super(Online_Classifier,self).__init__()
        if dim_in == -1:
            dim_in = 512 if model_name == 'wide_resnet28w8' else 128
        if not mlp:
            print(f"Using Linear classifier {dim_in}, {num_classes}")
            self.head = nn.Linear(dim_in, num_classes)
        else:
            print(f"Using {depth}-layer MLP Linear classifier {dim_in}, {num_classes}")
            list_layers = [nn.Linear(dim_in, dim_in), nn.ReLU(inplace=True)]
            for _ in range(depth-2):
                list_layers += [nn.Linear(dim_in, dim_in), nn.ReLU(inplace=True)]
            list_layers += [nn.Linear(dim_in, num_classes)]

            self.head = nn.Sequential(*list_layers)

    def forward(self,x):
        return self.head(x)

def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=True)


class WideBasicBlock(nn.Module):
    def __init__(
        self,
        in_planes,
        planes,
        dropout_rate,
        stride=1
    ):
        super(WideBasicBlock, self).__init__()
        self.bn1 = nn.BatchNorm2d(in_planes)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, padding=1, bias=True)
        self.dropout = nn.Dropout(p=dropout_rate)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=True),
            )

    def forward(self, x):
        out = self.dropout(self.conv1(F.relu(self.bn1(x))))
        out = self.conv2(F.relu(self.bn2(out)))
        out += self.shortcut(x)

        return out

class PawsWideResNet(nn.Module):

    def __init__(
        self,
        depth,
        widen_factor,
        drop_rate=0.3
    ):
        super(PawsWideResNet, self).__init__()

        self.in_planes = 16
        assert ((depth-4) % 6 == 0), 'Wide-resnet depth should be 6n+4'
        n = (depth-4)/6
        k = widen_factor

        nStages = [16, 16*k, 32*k, 64*k]

        self.conv1 = conv3x3(3, nStages[0])
        self.layer1 = self._wide_layer(WideBasicBlock, nStages[1], n, drop_rate, stride=1)
        self.layer2 = self._wide_layer(WideBasicBlock, nStages[2], n, drop_rate, stride=2)
        self.layer3 = self._wide_layer(WideBasicBlock, nStages[3], n, drop_rate, stride=2)
        self.bn1 = nn.BatchNorm2d(nStages[3], momentum=0.99)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.channels = nStages[3]

    def _wide_layer(self, block, planes, num_blocks, drop_rate, stride):
        strides = [stride] + [1]*(int(num_blocks)-1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, drop_rate, stride))
            self.in_planes = planes

        return nn.Sequential(*layers)

    def forward(self, x):
        x = self.conv1(x)
        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = F.relu(self.bn1(x))
        x = self.avgpool(x)
        x = x.view(x.size(0), -1)

        return x
