import argparse
import os
import shutil
import time
import random 
import numpy as np
from collections import OrderedDict

import torch
import torch.nn as nn
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.optim
import torch.utils.data
import torchvision.transforms as transforms
import torchvision.datasets as datasets
from torch.optim.lr_scheduler import StepLR, LambdaLR
from tensorboardX import SummaryWriter
import models.wrn as resnet
from models.wrn import BasicBlock, LambdaLayer

model_names = sorted(name for name in resnet.__dict__
    if name.islower() and not name.startswith("__")
                     and name.startswith("resnet")
                     and callable(resnet.__dict__[name]))

print(model_names)

parser = argparse.ArgumentParser(description='Propert ResNets for CIFAR10 in pytorch')
parser.add_argument('--arch', '-a', metavar='ARCH', default='resnet32',
                    choices=model_names,
                    help='model architecture: ' + ' | '.join(model_names) +
                    ' (default: resnet32)')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('--epochs', default=200, type=int, metavar='N',
                    help='number of total epochs to run')
parser.add_argument('--start-epoch', default=0, type=int, metavar='N',
                    help='manual epoch number (useful on restarts)')
parser.add_argument('-b', '--batch-size', default=128, type=int,
                    metavar='N', help='mini-batch size (default: 128)')
parser.add_argument('--dataset', default='cifar10', type=str,
                    help='dataset')
parser.add_argument('--lr', '--learning-rate', default=0.1, type=float,
                    metavar='LR', help='initial learning rate')
parser.add_argument('--lr_scheduler',  default='LLR', type=str,
                     help='lr scheduler')
parser.add_argument('--momentum', default=0.9, type=float, metavar='M',
                    help='momentum')
parser.add_argument('--weight-decay', '--wd', default=1e-4, type=float,
                    metavar='W', help='weight decay (default: 1e-4)')
parser.add_argument('--print-freq', '-p', default=50, type=int,
                    metavar='N', help='print frequency (default: 50)')
parser.add_argument('--resume', default='', type=str, metavar='PATH',
                    help='path to latest checkpoint (default: none)')
parser.add_argument('-e', '--evaluate', dest='evaluate', action='store_true',
                    help='evaluate model on validation set')
parser.add_argument('--pretrained', dest='pretrained', action='store_true',
                    help='use pre-trained model')
parser.add_argument('--half', dest='half', action='store_true',
                    help='use half-precision(16-bit) ')
parser.add_argument('--save-dir', dest='save_dir',
                    help='The directory used to save the trained models',
                    default='save_temp', type=str)
parser.add_argument('--save-every', dest='save_every',
                    help='Saves checkpoints at every specified number of epochs',
                    type=int, default=10)
parser.add_argument('--depth-penalty', '--dp', default=1e-3, type=float,
                    help='weight of depth penalty')
parser.add_argument('--retrain', default='', type=str, metavar='PATH',
                    help='path to retraining checkpoint (default: none)')
parser.add_argument('--gpu',
                    help='which gpu to use',
                    default=0, type=int)
parser.add_argument('--seed',
                    help='random seed',
                    default=2024, type=int)
parser.add_argument('--mode',
                    help='the mode of lambda',
                    default='constant', type=str)
best_prec1 = 0

# # Block number
# Block_num = {'resnet20': 9, 
#             'resnet26': 12,
#             'resnet32': 15,
#             'resnet38': 18,
#             'resnet44': 21,
#             'resnet56': 27,
#             'resnet110': 54,
#             'resnet1202': 600,
#             }

block_res20 = np.array([4608, 4608, 4608, 18432, 18432, 73728, 73728])
block_res26 = np.array([4608, 4608, 4608, 4608, 18432, 18432, 18432, 73728, 73728, 73728])
block_res26_4 = np.array([73728, 73728, 73728, 73728, 294912, 294912, 294912, 1179648, 1179648, 1179648])
block_res32 = np.array([4608, 4608, 4608, 4608, 4608, 18432, 18432, 18432, 18432, 73728, 73728, 73728, 73728])
block_res32_4 = np.array([73728, 73728, 73728, 73728, 73728, 294912, 294912, 294912, 294912, 1179648, 1179648, 1179648, 1179648])
block_res38 = np.array([4608, 4608, 4608, 4608, 4608, 4608, 18432, 18432, 18432, 18432, 18432, 73728, 73728, 73728, 73728, 73728])
block_res38_4 = np.array([73728, 73728, 73728, 73728, 73728, 73728, 294912, 294912, 294912, 294912, 294912, 1179648, 1179648, 1179648, 1179648, 1179648])
block_res44 = np.array([4608, 4608, 4608, 4608, 4608, 4608, 4608, 18432, 18432, 18432, 18432, 18432, 18432, 73728, 73728, 73728, 73728, 73728, 73728])
block_res56 = np.array([4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728])
block_res110 = np.array([4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 
                         18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 
                         73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728
                         ])
block_res164 = np.array([4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 4608, 
                          18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 18432, 
                          73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728, 73728
                        ])


norm_res26_4 = np.array([3.9, 6.12, 5.62, 6.13, 7.77, 7.68, 15.72, 25.0, 25.63, 23.01])
norm_res32_4 = np.array([3.42, 4.33, 4.7, 6.98, 5.91, 3.51, 4.72, 6.48, 13.82, 22.23, 22.34, 28.17, 21.48])
norm_res38_4 = np.array([5.82, 2.83, 3.9, 3.18, 5.5, 5.2, 3.94, 4.23, 6.19, 8.11, 13.68, 15.4, 16.57, 22.23, 28.33, 23.7])

norm_res56 = np.array([2.84, 5.0, 1.11, 4.73, 3.09, 4.85, 5.68, 5.64, 5.78, 5.08, 4.97, 6.32, 7.21, 7.84, 9.05, 12.56, 12.23, 18.62, 17.91, 16.8, 13.59, 11.89, 8.57, 8.71, 7.77])
norm_res110 = np.array([8.7, 7.21, 6.74, 6.34, 5.9, 6.19, 5.63, 5.94, 6.09, 5.58, 5.36, 4.92, 5.57, 4.95, 5.44, 4.97, 5.07, 5.3, 
                        7.38, 6.9, 7.37, 7.28, 7.01, 6.87, 7.62, 6.92, 6.89, 7.25, 7.0, 6.89, 7.25, 6.96, 6.7, 6.42, 6.33, 
                        11.12, 11.22, 11.06, 11.77, 11.54, 11.39, 11.61, 11.89, 11.33, 11.41, 11.38, 11.56, 11.16, 10.62, 9.99, 9.68, 8.64])
norm_res164 = np.array([2.51, 2.35, 1.87, 2.62, 1.6, 3.03, 2.22, 3.83, 2.1, 2.03, 1.0, 1.81, 1.77, 2.04, 1.62, 1.37, 1.91, 1.28, 1.14, 1.25, 1.84, 
                        4.01, 2.36, 3.31, 4.49, 2.54, 4.04, 1.31, 1.39, 2.72, 1.61, 1.68, 1.91, 1.83, 0.92, 1.52, 1.93, 1.4, 2.82, 2.75, 3.31, 4.52, 4.49, 5.21, 5.25, 5.35, 5.5, 5.88, 
                        8.46, 6.67, 8.67, 8.4, 10.48, 13.16, 13.06, 11.54, 11.24, 11.78, 10.9, 11.42, 10.78, 10.58, 9.84, 9.0, 9.37, 9.28, 8.59, 8.28, 8.27, 6.48, 6.82, 7.01, 7.13, 9.32, 7.17, 6.8, 7.12, 8.91, 11.8]
)

def get_weights(ini_weights):
    weights_root = np.sqrt(ini_weights)
    return weights_root / np.sum(weights_root)


def set_seed(seed):
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    torch.cuda.manual_seed(seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False

def lambda_block(l, mode, arch=None):
    if mode == 'constant':
        return 1
    elif mode == 'linear':
        return 2*l
        
    elif mode == 'ladder2':

        if arch == 'resnet26':
            block_weights = get_weights(block_res26)

        elif arch == 'resnet32':
            block_weights = get_weights(block_res32)

        elif arch == 'resnet38':
            block_weights = get_weights(block_res38)

        if arch == 'resnet26_4':
            norm_weights = norm_res26_4
            block_weights = get_weights(block_res26_4)

        elif arch == 'resnet32_4':
            norm_weights = norm_res32_4
            block_weights = get_weights(block_res32_4)

        elif arch == 'resnet38_4':
            norm_weights = norm_res38_4
            block_weights = get_weights(block_res38_4)

        elif arch == 'resnet56':
            norm_weights = norm_res56
            block_weights = get_weights(block_res56)

        elif arch == 'resnet110':
            norm_weights = norm_res110
            block_weights = get_weights(block_res110)

        elif arch == 'resnet164':
            norm_weights = norm_res164
            block_weights = get_weights(block_res164)

        if args.depth_penalty == 0:
            return 0
        else:
            return  float(block_weights[l-1]) / norm_weights[l-1] 

    elif mode == 'square':
        return 6 * l * l  
    elif mode == 'cubic':
        return 4 * l * l * l 

def main():
    global args, best_prec1
    args = parser.parse_args()

    set_seed(args.seed)
    device = torch.device('cuda:%d'%args.gpu)



    # Check the save_dir exists or not
    # args.save_dir = os.path.join(args.save_dir, 'dp_%.5f_seed_%d_mode_%s'%(args.depth_penalty, args.seed, args.mode))
    args.save_dir = '/'.join(args.retrain.split('/')[:-1])
    if not os.path.exists(args.save_dir):
        os.makedirs(args.save_dir)
    logs = open(os.path.join(args.save_dir, 'logs_retrain_lr_%s_%.5f_eps_%d.txt'%(args.lr_scheduler, args.lr, args.epochs)), 'w')
    for key, value in vars(args).items():
        logs.write(f"{key}: {value}\n")
    writer = SummaryWriter(os.path.join(args.save_dir))

    # model = torch.nn.DataParallel(resnet.__dict__[args.arch]())
    if args.dataset == 'cifar10':
        model = resnet.__dict__[args.arch]()
    elif args.dataset == 'cifar100':
        model = resnet.__dict__[args.arch](num_classes=100)
    model.to(device)

    # optionally retrain from a checkpoint
    if args.retrain:
        if os.path.isfile(args.retrain):
            print("=> loading checkpoint '{}'".format(args.retrain))
            logs.write("=> loading checkpoint '{}' \n".format(args.retrain))
            if args.gpu is None:
                checkpoint = torch.load(args.retrain)
            elif torch.cuda.is_available():
                # Map model to be loaded to specified single gpu.
                loc = 'cuda:{}'.format(args.gpu)
                checkpoint = torch.load(args.retrain, map_location=loc)
                new_checkpoint = OrderedDict()
                for k, v in checkpoint['state_dict'].items():
                    if 'module' in k:
                        k = k[7:]
                        new_checkpoint[k] = v
                    else:
                        new_checkpoint[k] = v

            # initial_params = model.state_dict()
            model.load_state_dict(new_checkpoint)

            s = 0
            # detact layer
            pruning_layer = []
            for name, module in model.named_modules():
            #     print(name)
            #     block_penalty = 0.0
                if isinstance(module, BasicBlock):
                    s +=1 
                    block_penalty = 0.0
                if isinstance(module, nn.Conv2d):
                    if s == 0:
                        continue
                    block_penalty += torch.sum(module.weight.pow(2))
                    if module.bias is not None:
                        block_penalty += torch.sum(module.bias.pow(2))     
                if 'shortcut' in name:
                    if isinstance(module, LambdaLayer):
                        s -= 1
                    else:
                        # depth_penalty += lambda_block(s, Block_num[args.arch], args.mode, args.arch) * torch.sqrt(block_penalty) 
                        if float(torch.sqrt(block_penalty)) < 0.5:
                            pruning_layer.append(name[:-9])

            
            for name, module in model.named_modules():
                if isinstance(module, nn.Conv2d) and name[:8] in pruning_layer:
                    # module.data = 0
                    # module.requires_grad = False
                    # 将权重和偏置设置为0
                    with torch.no_grad():  # 使用torch.no_grad()以避免影响梯度计算
                        module.weight.zero_()  # 使用.zero_()直接在原地将权重设置为0
                        if module.bias is not None:
                            module.bias.zero_()  # 同理，如果有偏置，也设置为0
                    # 停止计算这些参数的梯度
                    module.weight.requires_grad = False
                    if module.bias is not None:
                        module.bias.requires_grad = False

        else:
            print("=> no checkpoint found at '{}'".format(args.retrain))
            logs.write("=> no checkpoint found at '{}'\n".format(args.retrain))

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            print("=> loading checkpoint '{}'".format(args.resume))
            checkpoint = torch.load(args.resume)
            args.start_epoch = checkpoint['epoch']
            best_prec1 = checkpoint['best_prec1']
            model.load_state_dict(checkpoint['state_dict'])
            print("=> loaded checkpoint '{}' (epoch {})"
                  .format(args.evaluate, checkpoint['epoch']))
        else:
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

    if args.dataset == 'cifar10':
        train_loader = torch.utils.data.DataLoader(
            datasets.CIFAR10(root='./data', train=True, transform=transforms.Compose([
                transforms.RandomHorizontalFlip(),
                transforms.RandomCrop(32, 4),
                transforms.ToTensor(),
                normalize,
            ]), download=True),
            batch_size=args.batch_size, shuffle=True,
            num_workers=args.workers, pin_memory=True)

        val_loader = torch.utils.data.DataLoader(
            datasets.CIFAR10(root='./data', train=False, transform=transforms.Compose([
                transforms.ToTensor(),
                normalize,
            ])),
            batch_size=128, shuffle=False,
            num_workers=args.workers, pin_memory=True)
    
    elif args.dataset == 'cifar100':
        train_loader = torch.utils.data.DataLoader(
            datasets.CIFAR100(root='./data', train=True, transform=transforms.Compose([
                transforms.RandomHorizontalFlip(),
                transforms.RandomCrop(32, 4),
                transforms.ToTensor(),
                normalize,
            ]), download=True),
            batch_size=args.batch_size, shuffle=True,
            num_workers=args.workers, pin_memory=True)

        val_loader = torch.utils.data.DataLoader(
            datasets.CIFAR100(root='./data', train=False, transform=transforms.Compose([
                transforms.ToTensor(),
                normalize,
            ])),
            batch_size=128, shuffle=False,
            num_workers=args.workers, pin_memory=True)

    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda()

    if args.half:
        model.half()
        criterion.half()

    optimizer = torch.optim.SGD(model.parameters(), args.lr,
                                momentum=args.momentum,
                                weight_decay=args.weight_decay)

    # lr_scheduler = torch.optim.lr_scheduler.MultiStepLR(optimizer,
    #                                                     milestones=[100, 150], last_epoch=args.start_epoch - 1)

    if args.lr_scheduler == 'LLR':

        # 定义学习率调度器的lambda函数
        def lr_lambda(epoch, warm_up_epochs, total_epochs):
            if warm_up_epochs == 0:
                return max(0.0, 1.0 - epoch / total_epochs)
            else:
                if epoch < warm_up_epochs:
                    return (epoch + 1) / warm_up_epochs
                else:
                    # 线性递减到 0
                    return max(0.0, 1.0 - (epoch - warm_up_epochs+1) / (total_epochs +1  - warm_up_epochs))


        # 定义外部参数
        warm_up_epochs = int(args.epochs / 10)

        # 创建学习率调度器时传入外部参数
        scheduler = LambdaLR(optimizer, lr_lambda=lambda epoch: lr_lambda(epoch, warm_up_epochs, args.epochs))

    if args.arch in ['resnet1202', 'resnet110']:
        # for resnet1202 original paper uses lr=0.01 for first 400 minibatches for warm-up
        # then switch back. In this setup it will correspond for first epoch.
        for param_group in optimizer.param_groups:
            param_group['lr'] = args.lr*0.1

    
    if args.evaluate:
        validate(val_loader, model, criterion, logs, device)
        return

    for epoch in range(args.start_epoch, args.epochs):

        # train for one epoch
        print('current lr {:.5e}'.format(optimizer.param_groups[0]['lr']))
        logs.write('current lr {:.5e} \n'.format(optimizer.param_groups[0]['lr']))
        train(train_loader, model, criterion, optimizer, epoch, logs, writer, device)
        scheduler.step()

        # evaluate on validation set
        prec1 = validate(val_loader, model, criterion, logs, device)
        writer.add_scalar('Val Prec@1', prec1, epoch)

        # remember best prec@1 and save checkpoint
        is_best = prec1 > best_prec1
        best_prec1 = max(prec1, best_prec1)

        if epoch > 0 and epoch % args.save_every == 0 and is_best:
            save_checkpoint({
                'epoch': epoch + 1,
                'state_dict': model.state_dict(),
                'best_prec1': best_prec1,
            }, is_best, filename=os.path.join(args.save_dir, 'checkpoint_lr_%s_%.5f_eps_%d.th'%(args.lr_scheduler, args.lr, args.epochs)))

        save_checkpoint({
            'state_dict': model.state_dict(),
            'best_prec1': best_prec1,
        }, is_best, filename=os.path.join(args.save_dir, 'model_lr_%s_%.5f_eps_%d.th'%(args.lr_scheduler, args.lr, args.epochs)))

    logs.write('\n Best prec@1: %4f \n'%best_prec1)
    logs.close()


def train(train_loader, model, criterion, optimizer, epoch, logs, writer, device):
    """
        Run one train epoch
    """
    batch_time = AverageMeter()
    data_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()

    # switch to train mode
    model.train()

    end = time.time()
    for i, (input, target) in enumerate(train_loader):

        # measure data loading time
        data_time.update(time.time() - end)

        target = target.to(device)
        input_var = input.to(device)
        target_var = target
        if args.half:
            input_var = input_var.half()

        # compute output
        output = model(input_var)
        loss = criterion(output, target_var)

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        output = output.float()
        loss = loss.float()
        # measure accuracy and record loss
        prec1 = accuracy(output.data, target)[0]
        losses.update(loss.item(), input.size(0))
        top1.update(prec1.item(), input.size(0))

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if i % args.print_freq == 0:
            print('Epoch: [{0}][{1}/{2}]\t'
                  'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                  'Data {data_time.val:.3f} ({data_time.avg:.3f})\t'
                  'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                  'Prec@1 {top1.val:.3f} ({top1.avg:.3f}) '.format(
                      epoch, i, len(train_loader),  batch_time=batch_time,
                      data_time=data_time, loss=losses, top1=top1))
            logs.write('Epoch: [{0}][{1}/{2}]\t'
                  'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                  'Data {data_time.val:.3f} ({data_time.avg:.3f})\t'
                  'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                  'Prec@1 {top1.val:.3f} ({top1.avg:.3f}) \n'.format(
                      epoch, i, len(train_loader),  batch_time=batch_time,
                      data_time=data_time, loss=losses, top1=top1))
        writer.add_scalar('Train loss', losses.avg, epoch)
        writer.add_scalar('Train Prec@1', top1.avg, epoch)




def validate(val_loader, model, criterion, logs, device):
    """
    Run evaluation
    """
    batch_time = AverageMeter()
    losses = AverageMeter()
    top1 = AverageMeter()

    # switch to evaluate mode
    model.eval()

    end = time.time()
    with torch.no_grad():
        for i, (input, target) in enumerate(val_loader):
            target = target.to(device)
            input_var = input.to(device)
            target_var = target.to(device)

            if args.half:
                input_var = input_var.half()

            # compute output
            output = model(input_var)
            loss = criterion(output, target_var)

            output = output.float()
            loss = loss.float()

            # measure accuracy and record loss
            prec1 = accuracy(output.data, target)[0]
            losses.update(loss.item(), input.size(0))
            top1.update(prec1.item(), input.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                print('Test: [{0}/{1}]\t'
                      'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                      'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                      'Prec@1 {top1.val:.3f} ({top1.avg:.3f})'.format(
                          i, len(val_loader), batch_time=batch_time, loss=losses,
                          top1=top1))
                logs.write('Test: [{0}/{1}]\t'
                      'Time {batch_time.val:.3f} ({batch_time.avg:.3f})\t'
                      'Loss {loss.val:.4f} ({loss.avg:.4f})\t'
                      'Prec@1 {top1.val:.3f} ({top1.avg:.3f}) \n'.format(
                          i, len(val_loader), batch_time=batch_time, loss=losses,
                          top1=top1))

    print(' * Prec@1 {top1.avg:.3f}'
          .format(top1=top1))
    logs.write(' * Prec@1 {top1.avg:.3f} \n'
          .format(top1=top1))

    return top1.avg

def save_checkpoint(state, is_best, filename='checkpoint.pth.tar'):
    """
    Save the training model
    """
    torch.save(state, filename)

class AverageMeter(object):
    """Computes and stores the average and current value"""
    def __init__(self):
        self.reset()

    def reset(self):
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count


def accuracy(output, target, topk=(1,)):
    """Computes the precision@k for the specified values of k"""
    maxk = max(topk)
    batch_size = target.size(0)

    _, pred = output.topk(maxk, 1, True, True)
    pred = pred.t()
    correct = pred.eq(target.view(1, -1).expand_as(pred))

    res = []
    for k in topk:
        correct_k = correct[:k].view(-1).float().sum(0)
        res.append(correct_k.mul_(100.0 / batch_size))
    return res


if __name__ == '__main__':
    main()