function [thresh,cntR,sumR,cntP,sumP] = evaluation_bdry_image(inFile,gtFile, prFile, nthresh, maxDist, thinpb)
% [thresh,cntR,sumR,cntP,sumP] = boundaryPR_image(inFile,gtFile, prFile, nthresh, maxDist, thinpb)
%
% Calculate precision/recall curve.
%
% INPUT
%	inFile  : Can be one of the following:
%             - a soft or hard boundary map in image format.
%             - a collection of segmentations in a cell 'segs' stored in a mat file
%             - an ultrametric contour map in 'doubleSize' format, 'ucm2'
%               stored in a mat file with values in [0 1].
%             - an ultrametric contour map in 'imageSize' format, 'ucm'
%               stored in a mat file with values in [0 1].
%
%	gtFile	: File containing a cell of ground truth boundaries
%   prFile  : Temporary output for this image.
%	nthresh	: Number of points in PR curve.
%   MaxDist : For computing Precision / Recall.
%   thinpb  : option to apply morphological thinning on segmentation
%             boundaries.
%
% OUTPUT
%	thresh		Vector of threshold values.
%	cntR,sumR	Ratio gives recall.
%	cntP,sumP	Ratio gives precision.
%
%
% extended to run correctly without mex files and without thinning

if nargin<6, thinpb = 1; end
if nargin<5, maxDist = 0.0075; end
if nargin<4, nthresh = 99; end

[p,v,e]=fileparts(inFile);
if strcmp(e,'.mat'),
    load(inFile);
end

if exist('ucm2', 'var'),
    pb = double(ucm2(3:2:end, 3:2:end));
    clear ucm2;
elseif exist('ucm', 'var'),
    pb = double(ucm);
    clear ucm
elseif ~exist('segs', 'var')
    pb = double(imread(inFile))/255;
    if length(size(pb)) == 3
        pb = mean(pb, 3);
    end
end


groundTruth = load(gtFile, 'groundTruth');
groundTruthVar = groundTruth.groundTruth;
if isempty(groundTruthVar),
    error(' bad gtFile !');
end

if ~exist('segs', 'var')
    thresh = linspace(1/(nthresh+1),1-1/(nthresh+1),nthresh)';
else
    if nthresh ~= numel(segs)
        warning('Setting nthresh to number of segmentations');
        nthresh = numel(segs);
    end
    thresh = 1:nthresh; thresh=thresh';
end

% zero all counts
cntR = zeros(size(thresh));
sumR = zeros(size(thresh));
cntP = zeros(size(thresh));
sumP = zeros(size(thresh));

for t = 1:nthresh,

    if ~exist('segs', 'var')
        bmap = (pb>=thresh(t));
    else
        bmap = logical(seg2bdry(segs{t},'imageSize'));
    end

    % thin the thresholded pb to make sure boundaries are standard thickness
    if thinpb,
        bmap = double(bwmorph(bmap, 'thin', inf));    % OJO
    end
    if true
        maxDistPixel = maxDist * sqrt(sum(size(bmap) .^2));
        r = floor(maxDistPixel);
        [xx, yy] = meshgrid(-r:r, -r:r);
        filter = double(sqrt(xx.^2 + yy.^2) <= maxDistPixel);
        bmap_thick = imfilter(bmap, filter);
    end

    % accumulate machine matches, since the machine pixels are
    % allowed to match with any segmentation
    accP = zeros(size(bmap));

    % compare to each seg in turn
    for i = 1:numel(groundTruthVar),
        % compute the correspondence
        if false
            [match1,match2] = correspondPixels(bmap, double(groundTruthVar{i}.Boundaries), maxDist);
        else
            gt_thick = imfilter(groundTruthVar{i}.Boundaries, filter);
            match1 = bmap & gt_thick;
            match2 = bmap_thick & groundTruthVar{i}.Boundaries;
        end
        % accumulate machine matches
        accP = accP | match1;
        % compute recall
        cntR(t) = cntR(t) + sum(match2(:)>0);
        sumR(t) = sumR(t) + sum(groundTruthVar{i}.Boundaries(:));
    end

    % compute precision
    sumP(t) = sumP(t) + sum(bmap(:));
    cntP(t) = cntP(t) + sum(accP(:));

end

% output
fid = fopen(prFile,'w');
if fid==-1,
    error('Could not open file %s for writing.', prFile);
end
fprintf(fid,'%10g %10g %10g %10g %10g\n',[thresh cntR sumR cntP sumP]');
fclose(fid);
