"""Hyperparameter configuration."""

import ml_collections


def d(**kwargs):
  """Helper of creating a config dict."""
  return ml_collections.ConfigDict(initial_dictionary=kwargs)


seed = 0


def get_config():
  """Get the default hyperparameter configuration."""
  config = ml_collections.ConfigDict()
  config.train_data_config = d(
    seed=seed,
    data_spec="data/ppzee-split=train.npy",
    shuffle=True,
    batchsize=1000,    # should correct this to sth smaller like 2048
    mode='swrep',
    fixed_batch=False  # Use one batch from the empirical distribution.
  )
  config.eval_data_config = d(
    shuffle=False,
    data_spec="data/ppzee-split=test.npy",
    batchsize=1000,
    mode='once',
    fixed_batch=False
  )
  config.train_eval_config = d(
    # epochs=100,
    # steps_per_epoch=10000,
    num_train_steps=8000,
    substeps=1,
    num_eval_steps=None,  # Run eval by looping through the (finite) eval_ds.
    log_metrics_every_steps=50,
    checkpoint_every_steps=8000,
    eval_every_steps=500,
    seed=seed,
    log_imgs_every_steps=100,
    # warm_start="",
    lr_decay_last_steps_ratio=0.1
  )

  config.model_config = d(
    rd_lambda=100.0,
    distort_type='mse',
    # nu_support_size=128,
    nu_support_size=1000,
  )
  # Below optimizer_config not used, but still provide it for compliance with
  # the API of jax_experiment.Experiment.
  config.optimizer_config = d(
    name='adam',
    args=dict(),  # b1, b2, etc.
    learning_rate=1e-3,
    lr_decay=True,
    decay_type='inv_sqrt',
    decay_factor=0.1,
    # gradient_clip_norm=1.0,
  )
  config.ckpt_restore_dir = 'None'
  return config


def get_cfg_str(config):
  from collections import OrderedDict
  runname_dict = OrderedDict()
  runname_dict['n'] = config.model_config.nu_support_size
  runname_dict['rd_lambda'] = config.model_config.rd_lambda

  from common import utils
  return utils.config_dict_to_str(runname_dict)


def get_hyper():
  """
  Produce a list of flattened dicts, each containing a hparam configuration overriding the one in
  get_config(), corresponding to one hparam trial/experiment/work unit.
  :return:
  """
  from common import hyper
  ns = hyper.sweep('model_config.nu_support_size', [100, 1000, 4000])
  rd_lambdas = [100, 300, 1000, 3000, 10000, 30000]
  rd_lambdas = hyper.sweep('model_config.rd_lambda', rd_lambdas)
  hparam_cfgs = hyper.product(ns, rd_lambdas)
  return hparam_cfgs
