"""Hyperparameter configuration."""

import ml_collections


def d(**kwargs):
  """Helper of creating a config dict."""
  return ml_collections.ConfigDict(initial_dictionary=kwargs)


seed = 0


def get_config():
  """Get the default hyperparameter configuration."""
  config = ml_collections.ConfigDict()
  config.train_data_config = d(
    seed=seed,
    data_spec="data/dig=all-speaker=theo-split=train-stft.npy",
    shuffle=True,
    batchsize=20,
    mode='swrep',
    fixed_batch=False  # Use one batch from the empirical distribution.
  )
  config.eval_data_config = d(
    shuffle=False,
    data_spec="data/dig=all-speaker=theo-split=test-stft.npy",
    batchsize=20,  # Using 'all' would cause OOM.
    mode='once',
    fixed_batch=False
  )
  # # Use stochastic eval batches rather than going thru the whole ds.
  # config.eval_data_config = d(
  #   shuffle=True,
  #   data_spec="data/dig=all-speaker=theo-split=test-stft.npy",
  #   batchsize=50,  # Using 'all' would cause OOM.
  #   # mode='once',  # takes too long
  #   mode='repeat',
  #   fixed_batch=False
  # )
  config.train_eval_config = d(
    # epochs=100,
    # steps_per_epoch=10000,
    num_train_steps=8000,
    substeps=1,
    num_eval_steps=None,  # Run eval by looping through the (finite) eval_ds.
    log_metrics_every_steps=50,
    checkpoint_every_steps=6000,
    eval_every_steps=500,
    seed=seed,
    log_imgs_every_steps=100
    # warm_start="",
  )

  config.model_config = d(
    rd_lambda=0.1,
    distort_type='mse',
    # nu_support_size=128,
    nu_support_size=1000,
  )
  # Below optimizer_config not used, but still provide it for compliance with
  # the API of jax_experiment.Experiment.
  config.optimizer_config = d(
    name='adam',
    args=dict(),  # b1, b2, etc.
    learning_rate=1e-2,
    lr_decay=True,
    lr_decay_rate=1e-3,
    # gradient_clip_norm=1.0,
  )
  config.ckpt_restore_dir = 'None'
  return config


def get_cfg_str(config):
  from collections import OrderedDict
  runname_dict = OrderedDict()
  runname_dict['n'] = config.model_config.nu_support_size
  runname_dict['rd_lambda'] = config.model_config.rd_lambda

  from common import utils
  return utils.config_dict_to_str(runname_dict)


def get_hyper():
  """
  Produce a list of flattened dicts, each containing a hparam configuration overriding the one in
  get_config(), corresponding to one hparam trial/experiment/work unit.
  :return:
  """
  from common import hyper
  ns = hyper.sweep('model_config.nu_support_size', [100000, 200000])
  rd_lambdas = [0.3, 1, 3, 10, 30, 100, 300]
  rd_lambdas = hyper.sweep('model_config.rd_lambda', rd_lambdas)
  hparam_cfgs = hyper.product(ns, rd_lambdas)
  return hparam_cfgs
