"""Default Hyperparameter configuration."""

import ml_collections


def d(**kwargs):
  """Helper of creating a config dict."""
  return ml_collections.ConfigDict(initial_dictionary=kwargs)


seed = 0


def get_config():
  """Get the default hyperparameter configuration."""
  config = ml_collections.ConfigDict()
  config.train_data_config = d(
    seed=seed,
    data_spec="sphere",
    data_dim=2,
    batchsize=100,
    gaussian_noise_var=0.1,
    fixed_batch=False,
  )
  config.eval_data_config = dict(reuse_train=True)
  config.train_eval_config = d(
    num_train_steps=5000,
    num_eval_steps=1,
    log_metrics_every_steps=100,
    log_imgs_every_steps=100,
    checkpoint_every_steps=4000,
    eval_every_steps=100,  # In case training converged too fast.
    seed=seed,
    # warm_start="",
  )

  config.model_config = d(
    rd_lambda=10.0,
    latent_dim=10,
    num_samples=40000,  # NERDlagr.py
    data_dim=config.train_data_config.data_dim,
    distort_type='half_sse',
    scheduled_num_steps=config.train_eval_config.num_train_steps,
    transform_config=d(
      # encoder_units=[10, 10],
      decoder_units=[config.train_data_config.data_dim] * 2,  # NERD uses 2-layer MLP
      # encoder_activation='softplus',
      decoder_activation='relu',
      # decoder_activation=None,
      # posterior_type='gaussian',
      prior_type='std_gaussian',
      # ar_hidden_units=[10, 10],
      # ar_activation='softplus',
      # maf_stacks=2,
      # iaf_stacks=0
    ),
    optimizer_config=d(
      name='adam',
      # name='sgd',
      # lr_decay=True,
      # lr_decay_rate=1e-3,
      learning_rate=1e-2,
      # clip_norm=1.0,
    )
  )
  config.optimizer_config = d(
    name='sgd',
    learning_rate=1e-2,
    # gradient_clip_norm=1.0,
  )
  config.ckpt_restore_dir = 'None'
  return config


def get_cfg_str(config):
  from collections import OrderedDict
  runname_dict = OrderedDict()
  runname_dict['sig2'] = config.train_data_config.gaussian_noise_var
  runname_dict['ldim'] = config.model_config.latent_dim
  runname_dict['n'] = config.model_config.num_samples
  runname_dict['rd_lambda'] = config.model_config.rd_lambda
  runname_dict['tseed'] = config.train_eval_config.seed

  from common import utils
  return utils.config_dict_to_str(runname_dict, skip_falsy=False)


def get_hyper():
  """
  Produce a list of flattened dicts, each containing a hparam configuration overriding the one in
  get_config(), corresponding to one hparam trial/experiment/work unit.
  :return:
  """
  from common import hyper
  gaussian_noise_vars = [0.1]
  gaussian_noise_vars = hyper.sweep("train_data_config.gaussian_noise_var",
                                    gaussian_noise_vars)
  nu_support_sizes = [10, 100, 1000, 3000]
  nu_support_sizes = hyper.sweep("model_config.nu_support_size",
                                 nu_support_sizes)
  # rd_lambdas = [0.01, 0.03, 0.1, 0.3, 1., 3., 10.]
  rd_lambdas = [100, 30, 10, 3, 2, 1]
  rd_lambdas = hyper.sweep('model_config.rd_lambda', rd_lambdas)
  opts = hyper.sweep('optimizer_config.name', ['sgd', 'adam'])
  tseeds = hyper.sweep('train_eval_config.seed', list(range(5)))

  hparam_cfgs = hyper.product(gaussian_noise_vars, nu_support_sizes,
                              rd_lambdas, opts, tseeds)
  return hparam_cfgs
