import os
import csv

class CSVLogger:
    '''
    data logger for recording and saving data as a CSV file
    '''
    def __init__(self, file_path, column_names=None, append=False):
        self.file_path = file_path
        self.columns = column_names

        # validate file extension
        if os.path.splitext(file_path)[1].lower() != '.csv':
            raise ValueError("File path must have a .csv extension.")

        # handle file existence conflicts
        if os.path.exists(file_path):
            if not append:
                raise FileExistsError("File to be created already exists while append mode is disabled.")
        else:
            if append:
                raise FileNotFoundError("File to be appended not found: {}".format(file_path))
            else:
                if not os.path.exists(os.path.dirname(file_path)):
                    raise FileNotFoundError("The directory specified for creating file does not exists.")
                if column_names is not None:
                    self.add_row_from_list(column_names)

    def add_row(self, row_data):
        ''' add a row to the CSV file given either a list or dictionary '''
        if isinstance(row_data, dict):
            self.add_row_from_dict(row_data)
        elif isinstance(row_data, list):
            self.add_row_from_list(row_data)
        else:
            raise TypeError("row_data must be dict or list")

    def add_row_from_list(self, list_data):
        ''' add a row to the CSV file given a list of row data '''
        assert isinstance(list_data, list), "Invalid type for input row data, which is supposed to be in list type."

        if self.columns is not None and len(list_data) != len(self.columns):
            raise ValueError("Data length ({}) doesn't match columns ({}). Missing or extra data may be involved.".format(len(list_data), len(self.columns)))

        with open(self.file_path, 'a', newline='', encoding='utf-8') as f:
            csv.writer(f).writerow(list_data)

    def add_row_from_dict(self, dict_data, missing_value='-'):
        ''' add a row to the CSV file given a dictionary of row data '''
        assert isinstance(dict_data, dict), "Invalid type for input row data, which is supposed to be in dict type."
        
        if self.columns is None:
            raise RuntimeError("Columns must be defined for dictionary input.")

        extra_keys = set(dict_data.keys()) - set(self.columns)
        if extra_keys:
            raise KeyError("Unexpected keys: {}".format(', '.join(extra_keys)))

        list_data = [dict_data.get(key, missing_value) for key in self.columns]
        with open(self.file_path, 'a', newline='', encoding='utf-8') as f:
            csv.writer(f).writerow(list_data)

    def _clear_content(self):
        f = open(self.file_path, 'w', encoding='utf-8')
        f.truncate()
        f.close()

    def _read_rows(self):
        if not os.path.exists(self.file_path):
            return []
        with open(self.file_path, 'r', encoding='utf-8') as f:
            return list(csv.reader(f))

    def add_column(self, column_name, list_data):
        ''' add a column to the CSV file given a column name and a list of column data '''
        assert isinstance(column_name, str), "Invalid type for column name, which is supposed to be in str type."
        assert isinstance(list_data, list), "Invalid type for column data, which is supposed to be in list type."

        if self.columns is None:
            raise RuntimeError("Original columns must be defined to add new column.")
        if column_name in self.columns:
            raise ValueError("Column \"{}\" already exists.".format(column_name))

        old_rows = self._read_rows()
        assert len(old_rows) > 0, "File is empty."

        if len(list_data) != (len(old_rows) - 1):
            raise ValueError("Fail to append column: length of new values ({}) doesn't match existing data rows ({})!".format(len(list_data), len(old_rows) - 1))

        new_rows = [old_rows[0] + [column_name]]

        for i, row in enumerate(old_rows[1:]):
            new_rows.append(row + [list_data[i]])

        self.columns.append(column_name)

        self._clear_content()

        with open(self.file_path, 'w', newline='', encoding='utf-8') as f:
            csv.writer(f).writerows(new_rows)