import os
import yaml
import torch
import random
import numpy as np


def set_random_seed(seed, deterministic=True):
    ''' set random seeds for reproducibility with optional CUDA determinism '''
    random.seed(seed)
    np.random.seed(seed)
    torch.manual_seed(seed)
    
    # handle GPU seeds and determinism
    if torch.cuda.is_available():
        torch.cuda.manual_seed(seed)
        torch.cuda.manual_seed_all(seed)
        if deterministic:
            torch.backends.cudnn.deterministic = True
            torch.backends.cudnn.benchmark = False


def load_config(file_path):
    ''' load hyper-parameters and configuration settings from YAML file and return a dictionary '''
    if not os.path.isfile(file_path):
        raise FileNotFoundError("File not found: {}".format(file_path))

    if not (os.path.splitext(file_path)[1].lower() in ['.yml', '.yaml']):
        raise ValueError("Configuration file must have an extension of \".yml\" or \".yaml\".")

    with open(file_path, 'r', encoding='utf-8') as f:
        return yaml.safe_load(f)


def save_config(dict_data, file_path):
    ''' save hyper-parameters and configuration settings to YAML file given a dictionary'''
    if not isinstance(dict_data, dict):
        raise TypeError("Expected dictionary, got {}".format(type(dict_data)))

    if not os.path.exists(os.path.dirname(file_path)):
        raise FileNotFoundError("Directory for saving file not found: {}".format(os.path.dirname(file_path)))

    if not (os.path.splitext(file_path)[1].lower() in ['.yml', '.yaml']):
        raise ValueError("Configuration file must has an extension of \".yml\" or \".yaml\".")

    with open(file_path, 'w', encoding='utf-8') as f:
        yaml.safe_dump(dict_data, f, sort_keys=False)
