#' Assume the target is to minimize f(x), for x in [-M, M]^d within the
#' d-dimensional Euclidean space.
#'
#' ProGO_opt is the main algorithm for conducting global optimization
#'
#'
#'
#' This lssSampler() function simulates the minima distribution mk(x)
#' using the latent slice sampling (LSS).
#' @param targetFun: f(x), the target function to minimize.
#' @param d: dimension of the feasible space.
#' @param lower.bounds: vector with length d, lower bounds of x for each dimension.
#' @param upper.bounds: vector with length d, upper bounds of x for each dimension.
#' @param s.scale: parameter for latent slice sampler, default at 20.
#' @param k: the parameter for the minima distribution
#' @param n.sim: size of final samples.
#' @param n.burn: size of burn in samples.

### function for lss sampler
lssMdSampler <- function(targetFun = x2norm,
                         lower.bounds = lower.bounds,
                         upper.bounds = upper.bounds,
                         k = 5,
                         d = 2,
                         s.scale = 20,
                         n.sim = 200,
                         n.burn = 200,
                         debug = F,
                         x_init = rep(0, d),
                         upper_thre = 1e2,
                         lower_thre = -1e2,
                         fk = 0,
                         seed = 1) {
    myfun <- function(x) {
      # Define the function value within domain
      if (all(x >= lower.bounds) & all(x <= upper.bounds)) {
        val = -k * (targetFun(x) - fk)
        if (val > upper_thre) {
          return(exp(upper_thre))
        } else if (val < lower_thre) {
          return(exp(lower_thre))
        }
        else{
          return(exp(val))
        }
      } else{
        return(0)
      }
    }
    #' Initialization
    time.start = Sys.time()
    set.seed(34567 + seed)
    w = m = numeric(n.sim + n.burn)
    x = s = l = theta = matrix(0, nrow = n.sim + n.burn, ncol = d)
    m[1] = 1
    x[1, ] = x_init
    s[1, ] = rgamma(d, shape = 2, scale = s.scale)
    w[1] = runif(1, min = 0, max = myfun(x[1, ]))
    l[1, ] = runif(d, min = x[1, ] - 0.5 * s[1, ], max = x[1, ] + 0.5 * s[1, ])
    theta[1, ] = s.scale
    
    if (debug == T) {
      i = 1
      cat("x =", x[i, ], ",\n s =", s[i, ], ",\n w =", w[i], ",\n l =", l[i, ], sep = " ")
    }
    
    #' Define the Adaptive Rejection to update samples
    #' xt | xt-1
    adaptiveRejection <- function(a = l0 - s0 / 2,b = l0 + s0 / 2, w = w0, x.last = x0){
        #' For each iteration m sample the proposal
        x.star = runif(d, a, b)
        while (myfun(x.star) <= w) {
          for (j in 1:d) {
            if (x.star[j] < x.last[j]) {
              a[j] = max(a[j], x.star[j])
            } else{
              b[j] = min(b[j], x.star[j])
            }
          }
          x.star = runif(d, a, b)
        }
        return(list(x.star = x.star))
    }

    # Run MCMC for LSS
    for (t in 1:(n.sim + n.burn - 1)) {
      adap_rej = adaptiveRejection(
        a = l[t, ] - s[t, ] / 2,
        b = l[t, ] + s[t, ] / 2,
        w = w[t],
        x.last = x[t, ]
      )
      x[t + 1, ] = adap_rej$x.star
      w[t + 1] = runif(1, min = 0, max = myfun(x[t + 1, ]))
      s[t + 1, ] = rexp(d, rate = 1 / s.scale) + abs(l[t, ] - x[t + 1, ]) * 2
      l[t + 1, ] = runif(d, min = x[t + 1, ] - 0.5 * s[t + 1, ], max = x[t +
                                                                           1, ] + 0.5 * s[t + 1, ])
      if (debug == T) {
        cat("x=", x[t + 1, ], ",\n s=", s[t + 1, ], ",\n w=", w[t + 1], ",\n l=", l[t +
                                                                                      1, ], sep = " ")
      }
    }
    time.end = Sys.time()
    time.run.lss = time.end - time.start
    
    # Keep samples after discarding burn-in
    x.sample = x[-(1:n.burn), ]
    s.sample = s #[-(1:n.burn),]
    w.sample = w #[-(1:n.burn)]
    l.sample = l #[-(1:n.burn),]
    
    # result summary
    x.sample = as.matrix(x.sample, ncol = d)
    return(list(x.sample = x.sample, runtime = time.run.lss))
}


ProGO_opt <- function(f = ackley,
                     n_iter = 200,
                     k = 5,
                     lower.bounds = rep(-32.768, 20),
                     upper.bounds = rep(32.768, 20),
                     d = 20,
                     reg_tol = 1e-8,
                     x_init = rep(0,d), 
                     y_optim = 0,
                     x_optim = 0, # 0 for ackley, 1 for levy
                     debug = FALSE,
                     seed = 1) {
    # initialization
    iter = 1
    result = list()
    queries = list()
    regret = reg_tol + 1
    y.min = Inf
    for (iter in c(1:n_iter)) {
      start_time_progo = Sys.time()
      if (iter == 1){
        fk_iter = 0
      } else {
        fk_iter = y.min
      }
      y_lssMdSampler = lssMdSampler(
        k = k,
        targetFun = f,
        d = d,
        lower.bounds = lower.bounds,
        upper.bounds = upper.bounds,
        x_init = x_init,
        fk = fk_iter,
        seed = seed
      )
      end_time_progo = Sys.time()
      runtime_iter = difftime(end_time_progo, start_time_progo, units = "secs")
      x.sample = as.matrix(y_lssMdSampler$x.sample)
      y.values = apply(x.sample, 1, f)
      min_index = which.min(y.values)
      y.min = min(y.values)
      x.min = x.sample[min_index, ]
      regret = abs(y.min - y_optim)
      minima_regret = norm((x.min - x_optim), "2") / d
      
      result_optim = list(
        y.min = y.min,
        x.min = x.min,
        regret = regret,
        minima_regret = minima_regret,
        log_gap = log(regret),
        runtime = runtime_iter,
        k = k
      )
      result[[iter]] = result_optim
      best_id = which.min(sapply(result, "[", 1))
      queries[[iter]] = list(x = result[[best_id]][[2]], y = result[[best_id]][[1]])
      
      iter = iter + 1
      k = exp(1) * k
      if (debug) {
        print(iter)
        print(result)
      }
      if (regret < reg_tol) {
        cat("The regret has already achieved the targeted accuracy at ", reg_tol, ".", sep="")
        return (list(result = result, queries = queries))
      }
    }
    
    return (list(result = result, queries = queries))
  }

#' Ackley function
#' https://www.sfu.ca/~ssurjano/ackley.html
ackley <- function(xx,
                   a = 20,
                   b = 0.2,
                   c = 2 * pi)
{
  ##########################################################################
  #
  # ACKLEY FUNCTION
  #
  # Authors: Sonja Surjanovic, Simon Fraser University
  #          Derek Bingham, Simon Fraser University
  # Questions/Comments: Please email Derek Bingham at dbingham@stat.sfu.ca.
  #
  # Copyright 2013. Derek Bingham, Simon Fraser University.
  #
  # THERE IS NO WARRANTY, EXPRESS OR IMPLIED. WE DO NOT ASSUME ANY LIABILITY
  # FOR THE USE OF THIS SOFTWARE.  If software is modified to produce
  # derivative works, such modified software should be clearly marked.
  # Additionally, this program is free software; you can redistribute it
  # and/or modify it under the terms of the GNU General Public License as
  # published by the Free Software Foundation; version 2.0 of the License.
  # Accordingly, this program is distributed in the hope that it will be
  # useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  # of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  # General Public License for more details.
  #
  # For function details and reference information, see:
  # http://www.sfu.ca/~ssurjano/
  #
  ##########################################################################
  #
  # INPUTS:
  #
  # xx = c(x1, x2, ..., xd)
  # a = constant (optional), with default value 20
  # b = constant (optional), with default value 0.2
  # c = constant (optional), with default value 2*pi
  #
  ##########################################################################
  
  d <- length(xx)
  
  sum1 <- sum(xx ^ 2)
  sum2 <- sum(cos(c * xx))
  
  term1 <- -a * exp(-b * sqrt(sum1 / d))
  term2 <- -exp(sum2 / d)
  
  # For minimization
  y <- term1 + term2 + a + exp(1)
  return(y)
}
#' Levy function
#' https://www.sfu.ca/~ssurjano/levy.html
levy <- function(xx)
{
  ##########################################################################
  #
  # LEVY FUNCTION
  #
  # Authors: Sonja Surjanovic, Simon Fraser University
  #          Derek Bingham, Simon Fraser University
  # Questions/Comments: Please email Derek Bingham at dbingham@stat.sfu.ca.
  #
  # Copyright 2013. Derek Bingham, Simon Fraser University.
  #
  # THERE IS NO WARRANTY, EXPRESS OR IMPLIED. WE DO NOT ASSUME ANY LIABILITY
  # FOR THE USE OF THIS SOFTWARE.  If software is modified to produce
  # derivative works, such modified software should be clearly marked.
  # Additionally, this program is free software; you can redistribute it
  # and/or modify it under the terms of the GNU General Public License as
  # published by the Free Software Foundation; version 2.0 of the License.
  # Accordingly, this program is distributed in the hope that it will be
  # useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  # of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  # General Public License for more details.
  #
  # For function details and reference information, see:
  # http://www.sfu.ca/~ssurjano/
  #
  ##########################################################################
  #
  # INPUT:
  #
  # xx = c(x1, x2, ..., xd)
  #
  ##########################################################################
  
  d <- length(xx)
  w <- 1 + (xx - 1) / 4
  
  term1 <- (sin(pi * w[1])) ^ 2
  term3 <- (w[d] - 1) ^ 2 * (1 + 1 * (sin(2 * pi * w[d])) ^ 2)
  
  wi <- w[1:(d - 1)]
  sum <- sum((wi - 1) ^ 2 * (1 + 10 * (sin(pi * wi + 1)) ^ 2))
  
  y <- term1 + sum + term3
  return(y)
}

# Example code

d = 2
ackley_progo = ProGO_opt(f=ackley, n_iter = 30, lower.bounds = rep(-20, d), upper.bounds = rep(20, d), d=d, seed = 11, debug=TRUE)





